<?php

namespace FiloBlu\Label\Model\Translate;

use InvalidArgumentException;
use Magento\Framework\App\Area;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\State;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Locale\Resolver;
use Magento\Framework\Phrase\RendererInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Translation\Model\ResourceModel\StringUtils;
use Magento\Translation\Model\ResourceModel\StringUtilsFactory;

/**
 * @class Composite
 * @package FiloBlu\Label\Model\Translate
 */
class Composite extends \Magento\Framework\Phrase\Renderer\Composite implements RendererInterface
{

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;
    /**
     * @var State
     */
    protected $appState;
    /**
     * @var StoreManagerInterface
     */
    protected $_storeManager;
    /**
     * @var Resolver
     */
    protected $_localeResolver;
    /**
     * @var StringUtilsFactory
     */
    private $_resourceFactory;

    /**
     * @param RendererInterface[] $renderers
     * @throws InvalidArgumentException
     */
    public function __construct(
        array                 $renderers,
        State                 $appState,
        ScopeConfigInterface  $scopeConfig,
        StoreManagerInterface $storeManager,
        Resolver              $localeResolver,
        StringUtilsFactory    $resource
    ) {
        $this->appState = $appState;
        $this->scopeConfig = $scopeConfig;
        $this->_storeManager = $storeManager;
        $this->_localeResolver = $localeResolver;
        $this->_resourceFactory = $resource;

        foreach ($renderers as $renderer) {
            if (!$renderer instanceof RendererInterface) {
                throw new InvalidArgumentException(
                    sprintf('Instance of the phrase renderer is expected, got %s instead.', get_class($renderer))
                );
            }
        }
        $this->_renderers = $renderers;
    }


    /**
     * Render source text
     *
     * @param array $source
     * @param array $arguments
     * @return string
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function render(array $source, array $arguments = [])
    {
        $result = $source;
        foreach ($this->_renderers as $render) {
            $result[] = $render->render($result, $arguments);
        }

        $label_result = end($result);

        if ($this->appState->getAreaCode() != Area::AREA_ADMINHTML && $this->scopeConfig->getValue('dev/registration_inline/active')) {

            $locale = $this->_localeResolver->getLocale();
            $validStoreId = $this->_storeManager->getStore()->getId();
            $this->saveTranslate($result[0], $result[1], $locale, $validStoreId);

        }


        return $label_result;
    }

    /**
     * Save translation
     *
     * @param String $string
     * @param String $translate
     * @param null $locale
     * @param int|null $storeId
     * @return $this
     * @throws LocalizedException
     */
    public function saveTranslate($string, $translate, $locale = null, $storeId = null)
    {
        $string = htmlspecialchars_decode($string);
        /** @var $resource StringUtils */
        $resource = $this->_resourceFactory->create();
        $connection = $resource->getConnection();
        $table = $resource->getMainTable();
        $translate = htmlspecialchars($translate, ENT_QUOTES);

        if ($locale === null) {
            $locale = $this->_localeResolver->getLocale();
        }

        if ($storeId === null) {
            $storeId = $this->getStoreId();
        }

        $select = $connection->select()->from(
            $table,
            ['key_id', 'translate']
        )->where(
            'store_id = :store_id'
        )->where(
            'locale = :locale'
        )->where(
            'string = :string'
        )->where(
            'crc_string = :crc_string'
        );
        $bind = [
            'store_id' => $storeId,
            'locale' => $locale,
            'string' => $string,
            'crc_string' => crc32($string),
        ];

        if ($row = $connection->fetchRow($select, $bind)) {
//            if ($row['translate'] != $translate) {
//                $connection->update($table, ['translate' => $translate], ['key_id=?' => $row['key_id']]);
//            }
        } else {
            $connection->insert(
                $table,
                [
                    'store_id' => $storeId,
                    'locale' => $locale,
                    'string' => $string,
                    'translate' => $translate,
                    'crc_string' => crc32($string)
                ]
            );
        }

        return $this;
    }
}
