<?php

namespace FiloBlu\GtidUrlRewrite\Helper;

use FiloBlu\GtidUrlRewrite\Model\CategoryRewrite;
use FiloBlu\GtidUrlRewrite\Model\UrlRewrite\DbStorage;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DataObject;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollectionFactory;
use Magento\UrlRewrite\Model\UrlRewriteFactory;

class Data
{
    /**
     * @var UrlRewriteFactory
     */
    protected $urlRewriteFactory;
    /**
     * @var ResourceConnection
     */
    protected $connection;
    /**
     * @var CategoryRepositoryInterface
     */
    protected $categoryRepository;
    /**
     * @var UrlRewriteCollectionFactory
     */
    protected $urlRewriteCollection;
    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var CategoryRewrite
     */
    protected $categoryRewrite;

    /**
     * @param UrlRewriteFactory $urlRewriteFactory
     * @param UrlRewriteCollectionFactory $urlRewriteCollection
     * @param ResourceConnection $connection
     * @param CategoryRepositoryInterface $categoryRepository
     * @param StoreManagerInterface $storeManager
     * @param CategoryRewrite $categoryRewrite
     */
    public function __construct(
        UrlRewriteFactory $urlRewriteFactory,
        UrlRewriteCollectionFactory $urlRewriteCollection,
        ResourceConnection $connection,
        CategoryRepositoryInterface $categoryRepository,
        StoreManagerInterface $storeManager,
        CategoryRewrite $categoryRewrite
    )
    {
        $this->urlRewriteFactory = $urlRewriteFactory;
        $this->connection = $connection;
        $this->categoryRepository = $categoryRepository;
        $this->urlRewriteCollection = $urlRewriteCollection;
        $this->storeManager = $storeManager;
        $this->categoryRewrite = $categoryRewrite;
    }

    /**
     * @param $urls
     * @return void
     * @throws AlreadyExistsException
     * @throws UrlAlreadyExistsException
     */
    public function generateUrlRewrites($urls)
    {
        $currentCategory = $this->categoryRewrite->getCurrentCategory();
        $oldUrlKey = $currentCategory->getOrigData('url_key');
        $urlKey = $currentCategory->getUrlKey();

        $data = [];
        foreach ($urls as $url) {

            $parts = explode('/', rtrim($url->getRequestPath(), '.html'));
            $key = array_search($urlKey, $parts);

            if (is_numeric($key) && array_key_exists($key, $parts)) {
                $parts[$key] = $oldUrlKey;
            }

            $requestPath = implode('/', $parts) . '.html';

            $url->setRequestPath($requestPath);

            $oldUrlRewrite = $this->getUrlRewriteByData($url->getStoreId(), $currentCategory->getId(), $requestPath, $url->getRedirectType(), $url->getIsAutogenerated());

            if ($oldUrlRewrite) {
                $row = [
                    'request_path' => $requestPath,
                    'url_rewrite_id' => $oldUrlRewrite->getId()
                ];

                $this->updateUrlRewrite($row, 'request_path');
                continue;
            }

            $data[] = $url->toArray();
        }

        $this->insertMultiple($data);
    }

    /**
     * Insert multiple
     *
     * @param  array $data
     * @return void
     * @throws AlreadyExistsException|\Exception
     * @throws \Exception
     */
    protected function insertData($data): void
    {
        $connection = $this->connection->getConnection();
        foreach ($data as $item)
        {
            try {
                $connection->insert($connection->getTableName(DbStorage::TABLE_NAME), $item);
            } catch (\Exception $e) {

            }
        }
    }

    /**
     * Update url_rewrite entry using a direct query
     *
     * @param $data
     * @param $tableColumn
     * @return void
     * @throws UrlAlreadyExistsException
     */
    public function updateUrlRewrite($data, $tableColumn)
    {
        $connection = $this->connection->getConnection();
        //$connection->beginTransaction();

        $tableName = $connection->getTableName('url_rewrite');

        $bind = [
            $tableColumn => $data[$tableColumn]
        ];
        $where = [
            'url_rewrite_id = ?' => (int)$data['url_rewrite_id']
        ];

        try {
            $connection->update($tableName, $bind, $where);
           // $connection->commit();
        } catch (\Exception $e) {
           // $connection->rollBack();
            throw new UrlAlreadyExistsException(
                __('URL key for specified store already exists.'),
                $e,
                $e->getCode(),
                []
            );
        }
    }

    /**
     * Get current category url rewrite
     *
     * @param $category
     * @return DataObject|null
     * @throws NoSuchEntityException
     */
    public function getCurrentCategoryUrlRewrite($category)
    {
        $urlParts = [];

        $path = $category->getPath();
        $parts = explode('/', $path);
        unset($parts[0],$parts[1]);

        foreach ($parts as $categoryId)
        {
            $parentCategory = $this->categoryRepository->get($categoryId);

            if ($category->getId() == $parentCategory->getId()) {
                continue;
            }

            $urlParts[] = $parentCategory->getUrlKey();
        }

        $urlParts[] = $category->getOrigData('url_key');

        $path = implode('/', $urlParts) . ".html";

        return $this->getUrlRewriteByData(null,  $category->getId(), $path, 0, 0);
    }

    /**
     * Loads url rewrite by data array
     *
     * @param $storeId
     * @param $categoryId
     * @param $url
     * @param $redirectType
     * @param $isAutogenerated
     * @return DataObject|null
     */
    public function getUrlRewriteByData($storeId, $categoryId, $url = '', $redirectType = 301, $isAutogenerated = 1)
    {
        $urlRewriteCollection = $this->urlRewriteCollection->create();

        if (!empty($url)) {
            $urlRewriteCollection->addFieldToFilter('request_path', $url);
        }

        $urlRewriteCollection->addFieldToFilter('entity_id', $categoryId);
        $urlRewriteCollection->addFieldToFilter('entity_type', 'category');
        $urlRewriteCollection->addFieldToFilter('redirect_type', $redirectType);

        if (!empty($storeId)) {
            $urlRewriteCollection->addFieldToFilter('store_id', $storeId);
        }

        if (!empty($isAutogenerated)) {
            $urlRewriteCollection->addFieldToFilter('is_autogenerated', 1);
        }

        $urlRewrite = $urlRewriteCollection->getFirstItem();

        if ($urlRewrite->getId()) {
            return $urlRewrite;
        }

        return null;
    }

    /**
     * Generate url rewrite for newly creted category
     *
     * @param $urls
     * @return void
     * @throws AlreadyExistsException
     */
    public function generateNewUrlRewrites($urls)
    {
        $data = [];
        foreach ($urls as $url) {
            $data[] = $url->toArray();
        }

        $this->insertMultiple($data);
    }

    /**
     * Insert multiple
     *
     * @param  array $data
     * @return void
     * @throws AlreadyExistsException|\Exception
     * @throws \Exception
     */
    public function insertMultiple($data)
    {
        $connection = $this->connection->getConnection();

        try {
            $connection->insertMultiple($connection->getTableName(DbStorage::TABLE_NAME), $data);
        } catch (\Exception $e) {
            $x = null;
        }
    }

    /**
     * Prepare select statement for specific filter
     *
     * @param  array $data
     * @return Select
     */
    public function prepareSelect(array $data)
    {
        $connection = $this->connection->getConnection();
        $select = $connection->select();
        $select->from($connection->getTableName(DbStorage::TABLE_NAME));

        foreach ($data as $column => $value) {
            if (empty($value) && !is_numeric($value)) {
                $select->where($connection->quoteIdentifier($column) . ' is NULL');
                continue;
            }

            $select->where($connection->quoteIdentifier($column) . ' IN (?)', $value);
        }

        return $select;
    }

    /**
     * @param array $data
     * @return array
     */
    public function doFindAllByData(array $data)
    {
        $connection = $this->connection->getConnection();
        return $connection->fetchAll($this->prepareSelect($data));
    }

    /**
     * @param array $data
     * @return void
     */
    public function deleteByData(array $data)
    {
        $connection = $this->connection->getConnection();
        $connection->query(
            $this->prepareSelect($data)->deleteFromSelect($connection->getTableName(DbStorage::TABLE_NAME))
        );
    }
}
