<?php

namespace FiloBlu\GtidUrlRewrite\Model\UrlRewrite;

use FiloBlu\GtidUrlRewrite\Helper\Data;
use FiloBlu\GtidUrlRewrite\Model\CategoryRewrite;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\UrlRewrite\Model\OptionProvider;
use Magento\UrlRewrite\Model\Storage\AbstractStorage;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Magento\UrlRewrite\Service\V1\Data\UrlRewriteFactory;
use Psr\Log\LoggerInterface;

class DbStorage extends AbstractStorage
{
    /**
     * DB Storage table name
     */
    const TABLE_NAME = 'url_rewrite';

    /**
     * Code of "Integrity constraint violation: 1062 Duplicate entry" error
     */
    const ERROR_CODE_DUPLICATE_ENTRY = 1062;

    /**
     * @var AdapterInterface
     */
    protected $connection;

    /**
     * @var Resource
     */
    protected $resource;

    /**
     * @var LoggerInterface
     */
    private $logger;
    /**
     * @var CategoryRewrite
     */
    private $categoryRewrite;
    /**
     * @var Data
     */
    private $helper;

    protected $currentCategoryUrlRewrite = null;

    protected $currentCategory = null;

    /**
     * @param UrlRewriteFactory $urlRewriteFactory
     * @param DataObjectHelper $dataObjectHelper
     * @param ResourceConnection $resource
     * @param CategoryRewrite $categoryRewrite
     * @param Data $helper
     * @param LoggerInterface|null $logger
     */
    public function __construct(
        UrlRewriteFactory $urlRewriteFactory,
        DataObjectHelper $dataObjectHelper,
        ResourceConnection $resource,
        CategoryRewrite $categoryRewrite,
        Data $helper,
        LoggerInterface $logger = null
    ) {
        $this->connection = $resource->getConnection();
        $this->resource = $resource;
        $this->logger = $logger ?: ObjectManager::getInstance()
            ->get(LoggerInterface::class);

        $this->categoryRewrite = $categoryRewrite;
        $this->helper = $helper;
        parent::__construct($urlRewriteFactory, $dataObjectHelper);
    }



    /**
     * @param array $data
     * @return array
     */
    protected function doFindAllByData(array $data)
    {
        return $this->helper->doFindAllByData($data);
    }

    /**
     * @param array $urls
     * @return array|UrlRewrite[]
     * @throws \Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException
     */
    protected function doReplace(array $urls)
    {
        try {
            $category = $this->categoryRewrite->getCurrentCategory();
            $this->currentCategory = $category;

            $this->currentCategoryUrlRewrite = $this->helper->getCurrentCategoryUrlRewrite($category);

            $currentUrlKey = $category->getUrlKey();
            $oldUrlKey = $category->getOrigData('url_key');

            $canonicalUrls = array_filter($urls, function ($x) use ($category) {
                if ($x->getIsAutogenerated()) {
                    return $x->getRedirectType() == 0;
                }
            });

            if ($oldUrlKey) {
                $this->updateUrlRewrite($canonicalUrls, $currentUrlKey, $oldUrlKey);
                $nonCanonicalUrls = array_filter($urls, function ($x) {
                    return $x->getRedirectType() != 0;
                });

                if ($nonCanonicalUrls) {
                    $this->helper->generateUrlRewrites($nonCanonicalUrls);
                }
            } else {
                $this->helper->generateNewUrlRewrites($urls);
            }


        } catch (\Exception $e) {
            throw new \Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException(
                __('URL key for specified store already exists.'),
                $e,
                $e->getCode(),
                []
            );
        }
        return $urls;
    }

    /**
     * @param array $data
     * @return void
     */
    public function deleteByData(array $data)
    {
        $this->helper->deleteByData($data);
    }

    /**
     * @param array $data
     * @return array|mixed|null
     * @throws AlreadyExistsException
     */
    protected function doFindOneByData(array $data)
    {
        if (array_key_exists(UrlRewrite::REQUEST_PATH, $data)
            && is_string($data[UrlRewrite::REQUEST_PATH])
        ) {
            $result = null;

            unset($data[UrlRewrite::DESCRIPTION],$data[UrlRewrite::METADATA]);

            if ($this->currentCategory->getOrigData('url_path')) {
                //$data[UrlRewrite::TARGET_PATH] = $this->currentCategory->getUrlPath() . '.html';
                $requestPath = $this->currentCategory->getOrigData('url_path') . '.html';
                $data[UrlRewrite::REQUEST_PATH] = $requestPath;
            } else {
                $requestPath = $data[UrlRewrite::REQUEST_PATH];

                if ($data[UrlRewrite::REDIRECT_TYPE] == 0) {
                    unset($data[UrlRewrite::REQUEST_PATH]);
                } else {
                    $data[UrlRewrite::REQUEST_PATH] = $data[UrlRewrite::TARGET_PATH];
                    $data[UrlRewrite::TARGET_PATH] = $requestPath;
                }
            }

            $resultsFromDb = $this->connection->fetchAll($this->helper->prepareSelect($data));
            if ($resultsFromDb) {
                $urlRewrite = $this->extractMostRelevantUrlRewrite($requestPath, $resultsFromDb);
                $result = $this->prepareUrlRewrite($requestPath, $urlRewrite);
            }

            return $result;
        }

        return $this->connection->fetchRow($this->helper->prepareSelect($data));
    }

    /**
     * Extract most relevant url rewrite from url rewrites list
     *
     * @param  string $requestPath
     * @param  array  $urlRewrites
     * @return array|null
     */
    private function extractMostRelevantUrlRewrite(string $requestPath, array $urlRewrites): ?array
    {
        $prioritizedUrlRewrites = [];
        foreach ($urlRewrites as $urlRewrite) {
            $urlRewriteRequestPath = $urlRewrite[UrlRewrite::REQUEST_PATH];
            $urlRewriteTargetPath = $urlRewrite[UrlRewrite::TARGET_PATH];
            switch (true) {
                case rtrim($urlRewriteRequestPath, '/') === rtrim($urlRewriteTargetPath, '/'):
                    $priority = 99;
                    break;
                case $urlRewriteRequestPath === $requestPath:
                    $priority = 1;
                    break;
                case $urlRewriteRequestPath === urldecode($requestPath):
                    $priority = 2;
                    break;
                case rtrim($urlRewriteRequestPath, '/') === rtrim($requestPath, '/'):
                    $priority = 3;
                    break;
                case rtrim($urlRewriteRequestPath, '/') === rtrim(urldecode($requestPath), '/'):
                    $priority = 4;
                    break;
                default:
                    $priority = 5;
                    break;
            }
            $prioritizedUrlRewrites[$priority] = $urlRewrite;
        }
        ksort($prioritizedUrlRewrites);

        return array_shift($prioritizedUrlRewrites);
    }

    /**
     * Prepare url rewrite
     *
     * If request path matches the DB value or it's redirect - we can return result from DB
     * Otherwise return 301 redirect to request path from DB results
     *
     * @param  string $requestPath
     * @param  array  $urlRewrite
     * @return array
     */
    private function prepareUrlRewrite(string $requestPath, array $urlRewrite): array
    {
        $redirectTypes = [OptionProvider::TEMPORARY, OptionProvider::PERMANENT];
        $canReturnResultFromDb = (
            in_array($urlRewrite[UrlRewrite::REQUEST_PATH], [$requestPath, urldecode($requestPath)], true)
            || in_array((int) $urlRewrite[UrlRewrite::REDIRECT_TYPE], $redirectTypes, true)
        );
        if (!$canReturnResultFromDb) {
            $urlRewrite = [
                UrlRewrite::ENTITY_TYPE => 'category',
                UrlRewrite::ENTITY_ID => '0',
                UrlRewrite::REQUEST_PATH => $urlRewrite[UrlRewrite::REQUEST_PATH],
                UrlRewrite::TARGET_PATH => $requestPath,
                UrlRewrite::REDIRECT_TYPE => OptionProvider::PERMANENT,
                UrlRewrite::STORE_ID => $urlRewrite[UrlRewrite::STORE_ID],
                UrlRewrite::DESCRIPTION => null,
                UrlRewrite::IS_AUTOGENERATED => '0',
                UrlRewrite::METADATA => null,
            ];
        }

        return $urlRewrite;
    }

    /**
     * @param $urls
     * @param $requestPath
     * @param $oldRequestPath
     * @return void
     */
    public function updateUrlRewrite($urls, $requestPath, $oldRequestPath)
    {
        $emptyRewrites = [];

        foreach ($urls as $url) {
            $rewrite = $this->findOneByData($url->toArray());

            if (!$rewrite) {
                continue;
            }

            $urlRewriteId = $rewrite->getUrlRewriteId();

            if (empty($urlRewriteId)) {
                continue;
            }

            $rewritePath = $rewrite->getRequestPath();
            $tableColumn = 'request_path';

            if($rewrite->getRedirectType() != 0) {
                $rewritePath = $rewrite->getTargetPath();
                $tableColumn = 'target_path';
            }

            $urlParts = explode('/', rtrim( $rewritePath, '.html'));
            $key = array_search($oldRequestPath, $urlParts);

            if (!$key) {
                $keys = array_keys($urlParts);
                $key = end($keys);
            }

            $urlParts[$key] = $requestPath;

            $newRequestPath = implode("/", $urlParts);
            $newRequestPath .= '.html';

            $storeId = $rewrite->getStoreId();
            try {
                $data = [
                    $tableColumn => $newRequestPath,
                    'url_rewrite_id' => $urlRewriteId,
                    'store_id' => $storeId,
                ];
                $this->helper->updateUrlRewrite($data, $tableColumn);
            } catch (\Exception $exception) {
                continue;
            }
        }

        if (!empty($emptyRewrites)) {
            $this->helper->insertMultiple($emptyRewrites);
        }
    }
}
