<?php

declare(strict_types=1);

namespace FiloBlu\Flow\Model\From;

use FiloBlu\Flow\Helper\Product;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\Data\ProductLinkExtensionFactory;
use Magento\Catalog\Api\Data\ProductLinkExtensionInterface;
use Magento\Catalog\Api\Data\ProductLinkInterface;
use Magento\Catalog\Api\Data\ProductLinkInterfaceFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ProductFactory;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\StateException;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Throwable;
use Exception;
use http\Exception\RuntimeException;

/**
 * @class Grouped
 * @package FiloBlu\Flow\Model\From
 */
class Grouped extends AbstractFrom
{

    /**
     * @var ProductFactory
     */
    private $productFactory;
    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;
    /**
     * @var ProductLinkInterfaceFactory
     */
    private $productLinkFactory;
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;
    /**
     * @var ProductLinkExtensionFactory
     */
    private $productLinkExtensionFactory;

    private $connection;

    /**
     * @var Product
     */
    private $helperProduct;

    /**
     * @param Context $context
     * @param Registry $registry
     * @param ProductFactory $productFactory
     * @param ProductLinkInterfaceFactory $productLinkFactory
     * @param ProductLinkExtensionFactory $productLinkExtensionFactory
     * @param ProductRepositoryInterface $productRepository
     * @param ResourceConnection $resourceConnection
     * @param Product $helperProduct
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        Context                     $context,
        Registry                    $registry,
        ProductFactory              $productFactory,
        ProductLinkInterfaceFactory $productLinkFactory,
        ProductLinkExtensionFactory $productLinkExtensionFactory,
        ProductRepositoryInterface  $productRepository,
        ResourceConnection          $resourceConnection,
        Product                     $helperProduct,
        AbstractResource            $resource = null,
        AbstractDb                  $resourceCollection = null,
        array                       $data = []
    )
    {
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
        $this->productRepository = $productRepository;
        $this->productLinkFactory = $productLinkFactory;
        $this->resourceConnection = $resourceConnection;
        $this->productFactory = $productFactory;
        $this->productLinkExtensionFactory = $productLinkExtensionFactory;
        $this->helperProduct = $helperProduct;
        $this->connection = $this->resourceConnection->getConnection();
    }

    /**
     * @return void
     */
    public function _construct()
    {
        $this->_init(\FiloBlu\Flow\Model\ResourceModel\From\Grouped::class);
    }

    /**
     * @return void
     * @throws CouldNotSaveException
     * @throws InputException
     */
    public function processFileRows($file)
    {
        $mainTable = $this->connection->getTableName('flow_from_grouped');
        $selectGroupedSku = $this->connection->select()->from($mainTable, ['grouped_sku', 'grouped_name'])->where(
            'meta_file = :meta_file'
        )->distinct();
        $selectGroupedChildren = $this->connection->select()->distinct()->from($mainTable, ['child_sku', 'position', 'quantity','meta_id'])->where(
            'grouped_sku = :grouped_sku'
        )->where(
            'meta_file = :meta_file'
        );

        $groupedSkus = $this->connection->fetchPairs($selectGroupedSku, ['meta_file' => $file->getId()]);
        foreach ($groupedSkus as $groupedSku => $groupedName) {
            try {
                $grouped = $this->generateGrouped($groupedSku, $groupedName);
                $children = $this->connection->fetchAssoc($selectGroupedChildren, ['grouped_sku' => $groupedSku, 'meta_file' => $file->getId()]);
                $this->addChildren($grouped, $children);
            } catch (Exception $exception) {
                $this->markError($groupedSku, $file);
            } catch (Throwable $throwable) {
                $this->markError($groupedSku, $file);
            }
        }
    }

    public function process()
    {
        // TODO: Implement process() method.
    }

    /**
     * @param $groupedSku
     * @param $groupedName
     * @return ProductInterface
     * @throws CouldNotSaveException
     * @throws InputException
     * @throws StateException
     */
    public function generateGrouped($groupedSku, $groupedName): ProductInterface
    {
        try {
            $groupedProduct = $this->productRepository->get($groupedSku);
        } catch (NoSuchEntityException $e) {
            $groupedProduct = null;
        }

        if (isset($groupedProduct) && $groupedProduct->getTypeId() !== \Magento\GroupedProduct\Model\Product\Type\Grouped::TYPE_CODE) {
            throw new RuntimeException(
                sprintf('Product with SKU %s already exists but the type is not Grouped', $groupedProduct->getSku())
            );
        }

        if (!isset($groupedProduct)) {
            $groupedProduct = $this->productFactory->create();
            $groupedProduct
                ->setTypeId(
                    \Magento\GroupedProduct\Model\Product\Type\Grouped::TYPE_CODE)
                ->setAttributeSetId(4)
                ->setName($groupedName)
                ->setSku($groupedSku)
                ->setVisibility(Visibility::VISIBILITY_BOTH)
                ->setStatus(Status::STATUS_ENABLED)
                ->setStockData(['use_config_manage_stock' => 1, 'is_in_stock' => 1]);

            $groupedProduct = $this->productRepository->save($groupedProduct);
        }

        return $groupedProduct;
    }

    /**
     * @param ProductInterface $grouped
     * @param array $children
     * @return void
     * @throws CouldNotSaveException
     * @throws InputException
     * @throws NoSuchEntityException
     * @throws StateException
     */
    public function addChildren(ProductInterface $grouped, array $children)
    {
        $configData = $this->getChannel()->getChannelConfig()->getConfigData();
        $childVisibility = $configData['config']['grouped_child_visibility'];
        $childWebsite = false;
        $childCategoryIds = false;
        $childrenMetaId = [];

        $associated = $grouped->getProductLinks();
        foreach ($children as $sku => $child) {
            $key = $this->checkIfAlreadyAssociated($associated, $sku);

            $childProduct = $this->productFactory->create()->setStoreId(0)->loadByAttribute('sku', $sku);
            $childProduct->setVisibility($childVisibility);
            $childProduct->setStatus(1);

            try {
                $childProduct->save();
            }
            catch (Exception $e) {
                throw new RuntimeException('Error model from grouped: update children visibility');
            }

            if(count($childProduct->getWebsiteIds()) > 0) {
                $childWebsite = $childProduct->getWebsiteIds();
            }
            if(count($childProduct->getCategoryIds()) > 0) {
                $childCategoryIds = $childProduct->getCategoryIds();
            }

            if ($key === null) {
                $associated[] = $this->associateToGrouped($grouped, $child);
                $childrenMetaId[] = $child['meta_id'];
                continue;
            }

            if ($child['position'] !== null) {
                $associated[$key]->setPosition($child['position']);
            }

            if ($child['quantity'] !== null) {
                $associated[$key]
                    ->getExtensionAttributes()
                    ->setQty($child['quantity']);
            }

            if(!in_array($child['meta_id'], $childrenMetaId)) {
                $childrenMetaId[] = $child['meta_id'];
            }
        }

        $grouped->setProductLinks($associated);
        if($childWebsite) {
            $grouped->setWebsiteIds(array_diff($grouped->getWebsiteIds(), $grouped->getWebsiteIds()));
            $grouped->setWebsiteIds($childWebsite);
        }
        if($childCategoryIds) {
            $grouped->setCategoryIds($childCategoryIds);
        }

        $this->syncChildAttributesToGrouped($grouped, $child);

        try {
            $this->productRepository->save($grouped);
        }
        catch (Exception $e) {
            throw new RuntimeException('Error model from grouped: save grouped after add children');
        }

        if(count($childrenMetaId) > 0) {
            $this->connection->update(
                $this->connection->getTableName('flow_from_grouped'),
                [
                    'meta_processed' => 1
                ],
                'meta_id IN('.implode(",", $childrenMetaId).')'
            );
        }
    }

    /**
     * @param ProductInterface $grouped
     * @param $child
     * @return bool
     */
    public function syncChildAttributesToGrouped(ProductInterface $grouped, $child) {
        $attributesToCopy = $this->helperProduct->getSimpleAttributesSyncToGrouped();

        if (empty($attributesToCopy)) {
            return false;
        }

        $groupedProduct = $this->productFactory->create()->setStoreId(0)->load($grouped->getId());
        $storeIds = $groupedProduct->getStoreIds();
        array_unshift($storeIds, '0');
        foreach ($storeIds as $storeId) {
            $simple = $this->productFactory->create()->setStoreId($storeId)->loadByAttribute('sku', $child['child_sku']);
            foreach ($attributesToCopy as $attributeCode) {
                $attributesData = $simple->getData($attributeCode);
                if($attributesData) {
                    $groupedProduct->addAttributeUpdate($attributeCode, $attributesData, $storeId);
                }
            }
            unset($simple, $attributesData);
        }

        return true;
    }

    /**
     * @param $associatedArray
     * @param $childSku
     * @return int|string|null
     */
    public function checkIfAlreadyAssociated($associatedArray, $childSku)
    {
        foreach ($associatedArray as $key => $associated) {
            if ($childSku == $associated->getLinkedProductSku()) {
                return $key;
            }
        }

        return null;
    }

    /**
     * @param ProductInterface $grouped
     * @param array $child
     * @return ProductLinkInterface
     * @throws NoSuchEntityException
     */
    public function associateToGrouped(ProductInterface $grouped, array $child)
    {
        $product = $this->productRepository->get($child['child_sku']);
        $productLink = $this->productLinkFactory->create();

        $productLink
            ->setSku($grouped->getSku())
            ->setLinkType('associated')
            ->setLinkedProductSku($child['child_sku'])
            ->setLinkedProductType($product->getTypeId());
        if ($child['position'] !== null) {
            $productLink->setPosition($child['position']);
        }

        $productLinkExtension = $productLink->getExtensionAttributes();
        if ($productLinkExtension === null) {
            /** @var ProductLinkExtensionInterface $productLinkExtension */
            $productLinkExtension = $this->productLinkExtensionFactory->create();
        }
        $productLinkExtension->setQty($child['quantity'] ?? 1);
        $productLink->setExtensionAttributes($productLinkExtension);

        return $productLink;
    }

    /**
     * @param $file
     * @return bool
     */
    public function sendErrorNotifications($file)
    {
        return true;
    }

    /**
     * @param $groupedSku
     * @param $file
     * @return void
     */
    public function markError($groupedSku, $file)
    {
        $connection = $this->resourceConnection->getConnection();
        $mainTable = $connection->getTableName('flow_from_grouped');

        $connection->update($mainTable,
            ['meta_processed' => 2, 'meta_process_time' => date('Y-m-d H:i:s')],
            ['grouped_sku = ?' => $groupedSku, 'meta_file = ?' => $file->getId()]
        );
    }
}
