<?php
/**
 * Copyright © 2015 FiloBlu. All rights reserved.
 */

namespace FiloBlu\Flow\Model;

use Magento\Framework\Model\AbstractModel;

/**
 * Class Inboundflow
 * @package FiloBlu\Flow\Model
 * @method string getStatus()
 * @method integer getPriority()
 * @method string getFlow()
 * @method string getType()
 * @method integer getRetry()
 * @method string getChannel()
 * @method string getLog()
 * @method string getName()
 * @method self setRetry(integer $retry)
 * @method self setLog(string $log)
 * @method self setComment(string $comment)
 * @method self setName(string $name)
 * @method self setChannel(string $channel)
 * @method self setFlow(string $flow)
 * @method self setType(string $type)
 * @method self setPriority(integer $priority)
 * @method self setLastActivity(string $lastActivity)
 */
class Inboundflow extends AbstractModel
{

    /**
     * DEVELOPER SETTINGS JUST FOR DEBUG PURPOSE
     * Set to true to disable flow status changes
     * WARNING!!! REMEMBER TO SWITCH ON FALSE FOR PRODUCTION
     * @var boolean
     */
    const IS_DEBUG = false;

    /**
     * @var string
     */
    const STATUS_FOUND = 'found';

    /**
     * @var string
     */
    const STATUS_RECEIVED = 'received';

    /**
     * @var string
     */
    const STATUS_RECEIVING = 'receiving';

    /**
     * @var string
     */
    const STATUS_PARSED = 'parsed';

    /**
     * @var string
     */
    const STATUS_PARSING = 'parsing';

    /**
     * @var string
     */
    const STATUS_PROCESSED = 'processed';

    /**
     * @var string
     */
    const STATUS_PROCESSING = 'processing';

    /**
     * @var string
     */
    const STATUS_ERROR = 'error';

    /**
     * @var string
     */
    const STATUS_CLEANED = 'cleaned';

    /**
     * @var string
     */
    const STATUS_SUSPENDED = 'suspended';

    /**
     * @var string
     */
    const PRIORITY_HIGH = 'high';

    /**
     * @var string
     */
    const PRIORITY_LOW = 'low';

    /**
     * @var string
     */
    const PRIORITY_NORMAL = 'normal';

    /**
     * @var string
     */
    const PRIORITY_CRITICAL = 'critical';

    /**
     * @var string
     */
    const STATUS_DEAD = 'dead';

    /**
     * @return array
     */
    public static function getPriorityList()
    {
        return [
            self::PRIORITY_CRITICAL => 'CRITICAL',
            self::PRIORITY_HIGH => 'HIGH',
            self::PRIORITY_NORMAL => 'NORMAL',
            self::PRIORITY_LOW => 'LOW'
        ];
    }

    /**
     * @param array $prepend
     * @return array
     */
    public static function getStatusList($prepend = [])
    {
        return array_merge($prepend, [
            self::STATUS_FOUND => 'FOUND',
            self::STATUS_RECEIVING => 'RECEIVING',
            self::STATUS_RECEIVED => 'RECEIVED',
            self::STATUS_PARSED => 'PARSED',
            self::STATUS_PARSING => 'PARSING',
            self::STATUS_PROCESSED => 'PROCESSED',
            self::STATUS_PROCESSING => 'PROCESSING',
            self::STATUS_SUSPENDED => 'SUSPENDED',
            self::STATUS_ERROR => 'ERROR',
            self::STATUS_CLEANED => 'CLEANED',
            self::STATUS_DEAD => 'DEAD'
        ]);
    }

    /**
     * @param $priority
     * @return bool
     */
    public static function isValidPriority($priority)
    {
        switch ($priority) {
            case self::PRIORITY_CRITICAL:
            case self::PRIORITY_LOW:
            case self::PRIORITY_HIGH:
            case self::PRIORITY_NORMAL:
                return true;
        }

        return false;
    }

    /**
     * @param $status
     * @return bool
     */
    public static function isValidStatus($status)
    {
        switch ($status) {
            case self::STATUS_FOUND:
            case self::STATUS_RECEIVING:
            case self::STATUS_RECEIVED:
            case self::STATUS_PARSED:
            case self::STATUS_PARSING:
            case self::STATUS_PROCESSED:
            case self::STATUS_PROCESSING:
            case self::STATUS_ERROR:
            case self::STATUS_CLEANED:
            case self::STATUS_SUSPENDED:
            case self::STATUS_DEAD:
                return true;
        }

        return false;
    }

    /**
     * @return void
     * @noinspection MagicMethodsValidityInspection
     */
    public function _construct()
    {
        $this->_init(ResourceModel\Inboundflow::class);
    }

    /**
     * @param $status
     * @return Inboundflow
     */
    public function setStatus($status)
    {
        if (!self::IS_DEBUG) {
            parent::setData('status', $status);
        }
        return $this;
    }

    /**
     * @param int $errorCount
     * @return $this
     */
    public function setErrorCount($errorCount)
    {
        $this->setData('errors', (int)$errorCount);
        return $this;
    }

    /**
     * @return int
     */
    public function getErrorCount()
    {
        return (int)$this->getData('errors');
    }
}
