<?php
declare(strict_types=1);

namespace FiloBlu\Flow\Model\From;

use Exception;
use FiloBlu\Flow\Helper\SharedCatalogHelper;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\Store\Model\StoreManagerInterface;

/**
 * @Class SharedCatalog
 * @package FiloBlu\Flow\Model\From
 */
class SharedCatalog extends AbstractFrom
{

    const CUSTOM_SHARED_CATALOG_TYPE = 0;
    const SHARED_CATALOG_TABLE = 'flow_from_sharedcatalog';

    /**
     * @var mixed
     */
    private $sharedCatalogRepository;
    /**
     * @var SharedCatalogHelper
     */
    private $sharedCatalogHelper;
    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @param SharedCatalogHelper $sharedCatalogHelper
     * @param Context $context
     * @param Registry $registry
     * @param StoreManagerInterface $storeManager
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        SharedCatalogHelper   $sharedCatalogHelper,
        Context               $context,
        Registry              $registry,
        StoreManagerInterface $storeManager,
        AbstractResource      $resource = null,
        AbstractDb            $resourceCollection = null,
        array                 $data = []
    ) {
        $this->storeManager = $storeManager;
        $this->sharedCatalogHelper = $sharedCatalogHelper;
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
    }

    /**
     * @return SharedCatalog
     * @throws LocalizedException
     * @throws Exception
     */
    public function processFileRows($file)
    {
        if (!isset($this->sharedCatalogRepository)) {
            $this->sharedCatalogRepository = $this->sharedCatalogHelper->getSharedCatalogRepository();
        }

        $sharedCatalogRows = $this->sharedCatalogHelper->getInboundSharedCatalogRows($file->getId());

        foreach ($sharedCatalogRows as $sharedCatalogRow) {
            $sharedCatalog = $this->sharedCatalogHelper->checkIfSharedCatalogExists($sharedCatalogRow['name']);
            if ($sharedCatalog) {
                return $this;
            }

            if (!isset($this->sharedCatalogFactory)) {
                $this->sharedCatalogFactory = $this->sharedCatalogHelper->getSharedCatalogFactory();
            }

            $storeCode = $sharedCatalogRow['store_code'];

            if (!$storeCode) {
                $storeCode = 'admin';
            }

            $store = $this->storeManager->getStore($storeCode);
            if ($store->getId() == null) {
                // store not found TODO raise exception?
                return $this;
            }

            $taxClassId = $this->sharedCatalogHelper->getTaxClass($sharedCatalogRow['tax_class']);

            /** @var \Magento\SharedCatalog\Model\SharedCatalog $newSharedCatalog */
            $newSharedCatalog = $this->sharedCatalogFactory->create();
            $newSharedCatalog
                ->setType(self::CUSTOM_SHARED_CATALOG_TYPE)
                ->setName($sharedCatalogRow['name'])
                ->setDescription($sharedCatalogRow['description'])
                ->setStoreId($store->getId())
                ->setTaxClassId($taxClassId);

            $this->sharedCatalogHelper->getSharedCatalogRepository()->save($newSharedCatalog);
        }

        return $this;
    }

    public function process()
    {
        // TODO: Implement process() method.
    }

    public function sendErrorNotifications($file)
    {
        // TODO: Implement sendErrorNotifications() method.
    }

    /**
     * @return void
     */
    public function _construct()
    {
        $this->_init(\FiloBlu\Flow\Model\ResourceModel\From\SharedCatalog::class);
    }
}
