<?php

namespace FiloBlu\Flow\Model\Channel\In;

use Exception;
use FiloBlu\Flow\Api\DocumentInterfaceFactory;
use FiloBlu\Flow\Helper\LoggerProvider;
use FiloBlu\Flow\Model\Channel\ConfigFactory;
use Magento\App\Dir;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\Model\ResourceModel\IteratorFactory;
use Magento\Framework\ObjectManagerInterface;
use Zend_Db;
use Zend_Db_Statement_Exception;

/**
 * Class Document
 * @package FiloBlu\Flow\Model\Channel\In
 */
class Document extends AbstractModel
{

    /**
     * @var string
     */
    const STATUS_FOUND = 'found';

    /**
     * @var string
     */
    const STATUS_RECEIVED = 'received';

    /**
     * @var string
     */
    const STATUS_PARSED = 'parsed';

    /**
     * @var string
     */
    const STATUS_PROCESSED = 'processed';

    /**
     * @var string
     */
    const STATUS_ERROR = 'error';

    /**
     * @var string
     */
    const PRIORITY_HIGH = 'high';

    /**
     * @var string
     */
    const PRIORITY_LOW = 'low';
    /**
     * @var string
     */
    const PRIORITY_NORMAL = 'normal';
    /**
     * string
     */
    const DOCUMENT_FOLDER = 'documents';
    /**
     * @var int
     */
    protected $errorsProcessingDocuments = 0;
    /**
     * @var int
     */
    protected $documentToProcess = 0;
    /**
     * @var DocumentInterfaceFactory
     */
    protected $document;
    /**
     * @var DirectoryList
     */
    protected $dir;
    /**
     * @var IteratorFactory
     */
    protected $iteratorFactory;
    /**
     * @var
     */
    protected $channelData;
    /**
     * @var
     */
    protected $_connection;
    /**
     * @var ResourceConnection
     */
    private $_resourceConnection;

    /**
     * Document constructor.
     * @param LoggerProvider $loggerProvider
     * @param ConfigFactory $channelConfigFactory
     * @param ObjectManagerInterface $objectManager
     * @param DocumentInterfaceFactory $document
     * @param ResourceConnection $resourceConnection
     * @param DirectoryList $dir
     * @param IteratorFactory $iteratorFactory
     */
    public function __construct(
        LoggerProvider $loggerProvider,
        ConfigFactory $channelConfigFactory,
        ObjectManagerInterface $objectManager,
        DocumentInterfaceFactory $document,
        ResourceConnection $resourceConnection,
        DirectoryList $dir,
        IteratorFactory $iteratorFactory
    ) {
        parent::__construct($loggerProvider, $channelConfigFactory, $objectManager);
        $this->document = $document;
        $this->dir = $dir;
        $this->iteratorFactory = $iteratorFactory;
        $this->_resourceConnection = $resourceConnection;
    }

    /**
     * @return AdapterInterface
     */
    protected function getConnection()
    {
        if ($this->_connection) {
            return $this->_connection;
        }

        return ($this->_connection = $this->_resourceConnection->getConnection());
    }

    /**
     * @param $file
     * @return bool|mixed
     * @throws Exception
     */
    public function receiveFile($file)
    {
        if (!$this->_connector) {
            throw new Exception('no connector joined to channel');
        }

        $meta_file_id = $file->getId();

        $local_dir = $this->_connector->getLocalDir() . DIRECTORY_SEPARATOR . self::DOCUMENT_FOLDER . DIRECTORY_SEPARATOR . $meta_file_id;

        $this->_connector->setLocalDir($local_dir);

        $model = $this->document->create();

        $collection = $model->getCollection();
        $collection->addFieldToFilter('meta_processed', 0);
        $collection->addFieldToFilter('meta_file', $meta_file_id);
        $collection->addFieldToFilter('status', 'found');
        $collection->load();

        $result = true;

        foreach ($collection as $document) {
            try {
                $received = $this->_connector->receiveResource($document->getData('name'), null, FTP_BINARY, true, null);
                if ($received) {
                    $document->setData('status', 'received');
                    if ($document->getData('name') !== $received) {
                        $log = $document->getData('log');
                        $log = trim("Document name fixed from '" . $document->getData('name') . "' to '{$received}'" . $log . "\n");
                        $document->setData('log', $log);
                        $document->setData('name', $received);
                    }
                    $document->save();
                } else {
                    $document->setData('status', 'found');
                    $log = $document->getData('log');
                    $log .= "Document not received\n";
                    $document->setData('log', $log);
                    $document->save();
                    $result = false;
                }
            } catch (Exception $e) {
                $log = $document->getData('log');
                $log .= $e->getMessage() . "\n";
                $document->setData('log', $log);
                $document->save();
                $document = false;
                //$this->_logger->log('FTP',$e->getMessage());
            }
        }

        return $result;
    }

    /**
     * @param $file
     * @return bool
     * @throws Exception
     */
    public function parse($file)
    {
        if (!$this->_connector) {
            throw new Exception('no connector joined to channel');
        }

        $meta_file_id = $file->getId();

        /*$channel_config = $this->_connector->getConfig();
        $local_dir = $channel_config['local_dir'] . DIRECTORY_SEPARATOR . 'document' . DIRECTORY_SEPARATOR . $meta_file_id;*/

        $model = $this->document->create();

        $collection = $model->getCollection();
        $collection->addFieldToFilter('meta_processed', 0);
        $collection->addFieldToFilter('meta_file', $meta_file_id);
        $collection->addFieldToFilter('status', 'received');
        $collection->load();

        foreach ($collection as $document) {
            $document->setStatus('parsed');
            $document->save();
        }

        return true;
    }

    /**
     * @param $file
     * @param $channel
     * @return bool
     */
    public function insertData($file)
    {
        $meta_file_id = $file->getId();

        /** @var Document $model */
        $model = $this->document->create();

        $collection = $model->getCollection();
        $collection->addFieldToSelect(['id', 'name', 'destination']);
        $collection->addFieldToFilter('meta_file', $meta_file_id);
        $collection->addFieldToFilter('status', ['in' => [self::STATUS_PARSED, self::STATUS_ERROR]]);
        $collection->load();

        $this->documentToProcess = $collection->count();

        /** @var Iterator $iterator */
        $iterator = $this->iteratorFactory->create();
        $iterator->walk($collection->getSelect(), [[$this, 'documentProcess']], ['meta_file_id' => $meta_file_id]);

        if ($this->errorsProcessingDocuments > 0) {
            return false;
        }

        return true;
    }

    /**
     * @param $args
     * @return bool
     * @throws \Magento\Framework\Exception\FileSystemException
     */
    public function documentProcess($args)
    {
        $mediaDirectory = $this->dir->getPath('media');
        $varDirectory = $this->dir->getPath('var');

        $connector = $this->getChannelData()->getConnector();

        $file = $args['row']['name'];
        $destination = $args['row']['destination'];
        $row_id = $args['row']['id'];
        $source = $connector->getLocalDir();
        $meta_file_id = $args['meta_file_id'];
        $sourceFilePath = $source . DIRECTORY_SEPARATOR . self::DOCUMENT_FOLDER . DIRECTORY_SEPARATOR . $meta_file_id . DIRECTORY_SEPARATOR . $file;
        $destinationFilePath = $mediaDirectory . DIRECTORY_SEPARATOR . $destination . DIRECTORY_SEPARATOR . $file;

        if (file_exists($sourceFilePath)) {
            if (false === copy($sourceFilePath, $destinationFilePath)) {
                $this->setProcecessedStatus($row_id, self::STATUS_ERROR, "Unable to move {$sourceFilePath} -> {$destinationFilePath}");
                $this->errorsProcessingDocuments++;
            }
            else {
                $this->setProcecessedStatus($row_id, self::STATUS_PROCESSED);
            }
        } else {
            $this->setProcecessedStatus($row_id, self::STATUS_ERROR, "{$sourceFilePath} Does not exists");
            $this->errorsProcessingDocuments++;
        }

        return true;
    }

    /**
     * @param $document
     * @param string $status
     * @param string $reason
     */
    protected function setProcecessedStatus($row_id, $status = self::STATUS_PROCESSED, $reason = null)
    {
        $connection = $this->getConnection();
        $connection->update(
            $connection->getTableName('flow_document'),
            [
                'meta_processed' => 1,
                'status' => $status,
                'log' => $reason
            ],
            $connection->quoteInto('id = ?', $row_id)
        );
    }

    /**
     * @param $channel
     */
    public function setChannelData($channel){
        $this->channelData = $channel;
    }

    /**
     * @return mixed
     */
    public function getChannelData() {
        return $this->channelData;
    }

    /**
     * @return mixed
     */
    public function getDocumentToProcessCount(){
        return $this->documentToProcess;
    }

    /**
     * @return int
     */
    public function getProcessedDocumentErrorCount(){
        return $this->errorsProcessingDocuments;
    }
}
