<?php

namespace FiloBlu\Rma\Helper;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Sales\Api\Data\OrderPaymentInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order\Payment;
use Magento\Store\Model\ScopeInterface;

/**
 *
 */
class RefundParentOrderHelper
{
    const ONLINE_PAYMENT_REFUND_ENABLE = 'filoblu_disablerma_section/online_payment_refund_for_parent_order/enabled_online_payment_refund';
    const PAYMENT_METHOD_EXCLUDED_FOR_ONLINE_REFUND = 'filoblu_disablerma_section/online_payment_refund_for_parent_order/payment_methods_excluded';
    const HPP_METHOD_EXCLUDED_FOR_ONLINE_REFUND = 'filoblu_disablerma_section/online_payment_refund_for_parent_order/adyen_hpp_methods_excluded';
    const CHANGE_SIZE_PAY_VERSION_INDEX = 'change_size_pay_version';
    const CHANGE_SIZE_PAY_LATEST_VERSION = 2;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;

    public function __construct(
        ScopeConfigInterface     $scopeConfig,
        OrderRepositoryInterface $orderRepository
    )
    {
        $this->scopeConfig = $scopeConfig;
        $this->orderRepository = $orderRepository;
    }

    /**
     * @param int|null $storeId
     * @return bool
     */
    public function isOnlinePaymentRefundForParentEnabled(int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(self::ONLINE_PAYMENT_REFUND_ENABLE, ScopeInterface::SCOPE_STORE, $storeId);
    }
    
    /**
     * @param Payment $payment
     * @param Payment $parentOrderPayment
     * @param int|null $storeId
     * @return bool
     */
    public function isOrderAllowedToRefund(Payment $payment, Payment $parentOrderPayment, int $storeId = null): bool
    {
        $isAllowed = true;

        if ($payment->getAdditionalInformation(self::CHANGE_SIZE_PAY_VERSION_INDEX) === null ||
            (int)$payment->getAdditionalInformation(self::CHANGE_SIZE_PAY_VERSION_INDEX) < self::CHANGE_SIZE_PAY_LATEST_VERSION) {
            return false;
        }

        $paymentMethodsExcludedValue = $this->scopeConfig->getValue(self::PAYMENT_METHOD_EXCLUDED_FOR_ONLINE_REFUND, ScopeInterface::SCOPE_STORE, $storeId);
        if ($paymentMethodsExcludedValue === null) {
            return $isAllowed;
        }

        $paymentMethodsExcluded = explode(',', $paymentMethodsExcludedValue);
        if (in_array($parentOrderPayment->getMethod(), $paymentMethodsExcluded)) {
            if ($parentOrderPayment->getMethod() === 'adyen_hpp') {
                $ccType = $parentOrderPayment->getCcType() === null ? '' : $parentOrderPayment->getCcType();
                $isAllowed = $this->isAdyenHppAllowedToRefund($ccType, $storeId);
            } else {
                $isAllowed = false;
            }
        }

        return $isAllowed;
    }

    /**
     * @param string $ccType
     * @param int|null $storeId
     * @return bool
     */
    public function isAdyenHppAllowedToRefund(string $ccType, int $storeId = null): bool
    {
        $isAllowed = true;

        $hppMethodsExcludedValue = $this->scopeConfig->getValue(self::HPP_METHOD_EXCLUDED_FOR_ONLINE_REFUND, ScopeInterface::SCOPE_STORE, $storeId);
        if ($hppMethodsExcludedValue === null) {
            return true;
        }

        $hppMethodsExcluded = explode(',', $hppMethodsExcludedValue);
        if (in_array($ccType, $hppMethodsExcluded)) {
            $isAllowed = false;
        }

        return $isAllowed;
    }

    /**
     * @param OrderPaymentInterface $payment
     * @return OrderPaymentInterface|null
     */
    public function getParentOrderPayment(OrderPaymentInterface $payment)
    {
        $parentOrderId = $payment->getAdditionalInformation('parent_order_id');
        return $this->orderRepository->get($parentOrderId)
                                     ->getPayment();
    }

}
