<?php

namespace FiloBlu\ExtInventory\Model\Algorithms;

use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\InventoryApi\Api\GetSourcesAssignedToStockOrderedByPriorityInterface;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterfaceFactory;
use Magento\InventorySourceSelectionApi\Model\GetInStockSourceItemsBySkusAndSortedSource;
use Magento\InventorySourceSelectionApi\Model\GetSourceItemQtyAvailableInterface;
use FiloBlu\ExtInventory\Helper\Data as HelperData;

/**
 * Class FinalSourceBasedAlgorithm
 * @package FiloBlu\ExtInventory\Model\Algorithms
 */
class DistanceFromShippingAddressAlgorithm extends AbstractAlgorithm
{
    /**
     * @var SourceSelectionResultInterfaceFactory
     */
    private $sourceSelectionResultFactory;
    /**
     * @var GetSourceItemQtyAvailableInterface
     */
    private $getSourceItemQtyAvailable;
    /**
     * @var GetInStockSourceItemsBySkusAndSortedSource
     */
    private $getInStockSourceItemsBySkusAndSortedSource;
    /**
     * @var SourceSelectionItemInterfaceFactory
     */
    private $sourceSelectionItemFactory;
    /**
     * @var GetSourcesAssignedToStockOrderedByPriorityInterface
     */
    private $getSourcesAssignedToStockOrderedByPriority;
    /**
     * @var HelperData
     */
    private $helperData;

    /**
     * DistanceFromShippingAddressAlgorithm constructor.
     * @param SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory
     * @param GetSourceItemQtyAvailableInterface $getSourceItemQtyAvailable
     * @param GetInStockSourceItemsBySkusAndSortedSource $getInStockSourceItemsBySkusAndSortedSource
     * @param SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory
     * @param GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority
     * @param HelperData $helperData
     */
    public function __construct(
        SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory,
        GetSourceItemQtyAvailableInterface $getSourceItemQtyAvailable,
        GetInStockSourceItemsBySkusAndSortedSource $getInStockSourceItemsBySkusAndSortedSource,
        SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory,
        GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority,
        HelperData $helperData
    ) {
        $this->sourceSelectionResultFactory = $sourceSelectionResultFactory;
        $this->getSourceItemQtyAvailable = $getSourceItemQtyAvailable;
        $this->getInStockSourceItemsBySkusAndSortedSource = $getInStockSourceItemsBySkusAndSortedSource;
        $this->sourceSelectionItemFactory = $sourceSelectionItemFactory;
        $this->getSourcesAssignedToStockOrderedByPriority = $getSourcesAssignedToStockOrderedByPriority;
        $this->helperData = $helperData;
    }

    /**
     * @param InventoryRequestInterface $inventoryRequest
     * @return SourceSelectionResultInterface
     * @throws InputException
     * @throws LocalizedException
     */
    public function execute(InventoryRequestInterface $inventoryRequest): SourceSelectionResultInterface
    {
        $order = $this->getOrder($inventoryRequest);

        $countryId = $order->getShippingAddress()->getCountryId();
        $sourcesOptions = $this->getSources($inventoryRequest->getStockId(), $countryId, $inventoryRequest);

        if ($this->helperData->isDebugLoggingEnabled()){
            $writer = new \Zend\Log\Writer\Stream(BP . '/var/log/ext-inventory.log');
            $logger = new \Zend\Log\Logger();
            $logger->addWriter($writer);
            foreach ($sourcesOptions as $key => $option) {
                if (!isset($option['items'])){
                    continue;
                }
                foreach ($option['items'] as $sku => $data) {
                    $time = date("Y-m-d H:i:s");
                    $logger->info("{$time} || ORDER: {$order->getIncrementId()} | SOLUTION: {$key} | SKU: {$sku} | QTY DEDUCT: {$data->getQtyToDeduct()} | QTY AVAILABLE: {$data->getQtyAvailable()} | SOURCE: {$data->getSourceCode()} | FULL: {$option['fullOrder']}");
                }
            }
        }

        foreach ($sourcesOptions as $option) {
            if ($option['fullOrder']) {
                return $this->sourceSelectionResultFactory->create(
                    [
                        'sourceItemSelections' => array_values($option['items']),
                        'isShippable'          => true
                    ]
                );
            }
        }

        $itemsLeft = [];
        foreach ($inventoryRequest->getItems() as $item) {
            $itemsLeft[$item->getSku()] = (float)$item->getQty();
        }

        // no full order create mixed order
        $selections = [];
        foreach ($sourcesOptions as $option) {
            /**
             * @var string $sku
             * @var SourceSelectionItemInterface $a
             */
            foreach ($option['items'] as $sku => $a) {
                if ($itemsLeft[$sku] <= 0.0) {
                    continue;
                }

                $qtyToDeduct = min($itemsLeft[$sku], $a->getQtyAvailable());

                $selections[] = $this->sourceSelectionItemFactory->create([
                    'sourceCode'   => $a->getSourceCode(),
                    'sku'          => $a->getSku(),
                    'qtyToDeduct'  => $qtyToDeduct,
                    'qtyAvailable' => $a->getQtyAvailable()
                ]);
                $itemsLeft[$sku] -= $qtyToDeduct;
            }
        }

        return $this->sourceSelectionResultFactory->create(
            [
                'sourceItemSelections' => $selections,
                'isShippable'          => true
            ]
        );
    }

    /**
     * @param $stockId
     * @param $countryId
     * @param $inventoryRequest
     * @return array
     * @throws InputException
     * @throws LocalizedException
     */
    protected function getSources($stockId, $countryId, $inventoryRequest)
    {
        $output = [];
        $sku = [];

        foreach ($inventoryRequest->getItems() as $item) {
            $sku[$item->getSku()] = (float)$item->getQty();
        }

        $sources = $this->getSourcesAssignedToStockOrderedByPriority->execute($stockId);

        // There is one source that is located to the country where this order will be shipped to?
        $source_at_destination_country = false;
        foreach ($sources as $source) {
            if ($source->getCountryId() === $countryId){
                $source_at_destination_country = true;
                break;
            }
        }

        // If I have one source to the destination country
        // I need to process this backwards because of the array_unshift
        // at the end of this function.
        // The highest priority source need to be pushed on top of the resultset
        // at the end of the foreach
        if ($source_at_destination_country){
            $sources = array_reverse($sources);
        }

        foreach ($sources as $source) {
            $itemsInSource = $this->getInStockSourceItemsBySkusAndSortedSource->execute(array_keys($sku), [$source->getSourceCode()]);

            //Fix $itemsInSource with the real quantities
            if($inventoryRequest->getExtensionAttributes()->getUseReservations()){
                $order = $this->getOrder($inventoryRequest);
                $website_code = $order->getStore()->getWebsite()->getCode();
                $realQuantities = $this->helperData->getSalableQty(array_keys($sku),$website_code,$source->getSourceCode(),false);
                foreach ($itemsInSource as $sourceItem){
                    $realQty = (isset($realQuantities[$sourceItem->getSku()]) ? (int)$realQuantities[$sourceItem->getSku()] : 0);
                    $sourceItem->setQuantity($realQty);
                }
            }

            $result = [
                'source'     => $source->getSourceCode(),
                'country'    => $source->getCountryId(),
                'selections' => [],
                'fullOrder'  => count($sku) === count($itemsInSource)
            ];

            foreach ($itemsInSource as $stockItem) {
                $currentSku = $stockItem->getSku();

                if ($stockItem->getQuantity() < $sku[$currentSku]) {
                    $result['items'][$currentSku] = $this->sourceSelectionItemFactory->create([
                        'sourceCode'   => $source->getSourceCode(),
                        'sku'          => $currentSku,
                        'qtyToDeduct'  => $stockItem->getQuantity(),
                        'qtyAvailable' => $stockItem->getQuantity()
                    ]);
                    $result['fullOrder'] = false;
                } else {
                    $result['items'][$currentSku] = $this->sourceSelectionItemFactory->create([
                        'sourceCode'   => $source->getSourceCode(),
                        'sku'          => $currentSku,
                        'qtyToDeduct'  => $sku[$currentSku],
                        'qtyAvailable' => $sku[$currentSku]
                    ]);
                }
            }

            if ($source->getCountryId() === $countryId) {
                array_unshift($output, $result);
                continue;
            }
            $output[] = $result;
        }

        return $output;
    }
}
