<?php

namespace FiloBlu\ExtInventory\Helper;

use Exception;
use FiloBlu\ExtInventory\Api\InventoryFbOrderItemSourceRepositoryInterface;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\ProductRepository;
use Magento\CatalogInventory\Model\Stock;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryConfigurationApi\Api\GetStockItemConfigurationInterface;
use Magento\InventoryConfigurationApi\Exception\SkuIsNotAssignedToStockException;
use Magento\InventorySales\Model\ResourceModel\StockIdResolver;
use Magento\InventorySalesApi\Api\Data\SalesChannelInterface;
use Magento\InventorySalesApi\Api\IsProductSalableInterface;
use Magento\InventorySalesApi\Api\StockResolverInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\OrderItemExtension;
use Magento\Sales\Api\Data\OrderItemExtensionFactory;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Store\Api\StoreRepositoryInterface;
use Magento\Store\Model\ResourceModel\Website\CollectionFactory;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Zend_Db_Expr;
use Zend_Db_Statement_Exception;

/**
 * Class Data
 * @package FiloBlu\ExtInventory\Helper
 */
class Data extends AbstractHelper
{
    const XML_MSI_GENERAL_INVENTORY_TYPE = 'msi/general/inventory_type';
    /**
     * @var StoreManagerInterface
     */
    public $storeManager;
    /**
     * @var OrderRepositoryInterface
     */
    public $orderRepositoryInterface;
    /**
     * @var IsProductSalableInterface
     */
    protected $isProductSalableInterface;
    /**
     * @var StockResolverInterface
     */
    protected $stockResolver;
    /**
     * @var ProductRepository
     */
    protected $productRepository;
    /**
     * @var CollectionFactory
     */
    protected $websiteCollectionFactory;
    /**
     * @var StockIdResolver
     */
    protected $stockIdResolver;
    /**
     * @var StoreRepositoryInterface
     */
    protected $storeRepositoryInterface;
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;
    /** @var */
    private $reservedQuantityBySourceSelect;
    /**
     * @var GetStockItemConfigurationInterface
     */
    private $getStockItemConfiguration;
    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;
    /**
     * @var OrderItemExtensionFactory
     */
    private $orderItemExtensionFactory;
    /**
     * @var InventoryFbOrderItemSourceRepositoryInterface
     */
    private $inventoryFbOrderItemSourceRepository;

    /**
     * Data constructor.
     * @param Context $context
     * @param IsProductSalableInterface $isProductSalableInterface
     * @param StockResolverInterface $stockResolver
     * @param CollectionFactory $websiteCollectionFactory
     * @param ProductRepository $productRepository
     * @param StockIdResolver $stockIdResolver
     * @param StoreManagerInterface $storeManager
     * @param OrderRepositoryInterface $orderRepositoryInterface
     * @param StoreRepositoryInterface $storeRepositoryInterface
     * @param ResourceConnection $resourceConnection
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param GetStockItemConfigurationInterface $getStockItemConfiguration
     * @param OrderItemExtensionFactory $orderItemExtensionFactory
     * @param InventoryFbOrderItemSourceRepositoryInterface $inventoryFbOrderItemSourceRepository
     */
    public function __construct(
        Context                                       $context,
        IsProductSalableInterface                     $isProductSalableInterface,
        StockResolverInterface                        $stockResolver,
        CollectionFactory                             $websiteCollectionFactory,
        ProductRepository                             $productRepository,
        StockIdResolver                               $stockIdResolver,
        StoreManagerInterface                         $storeManager,
        OrderRepositoryInterface                      $orderRepositoryInterface,
        StoreRepositoryInterface                      $storeRepositoryInterface,
        ResourceConnection                            $resourceConnection,
        SearchCriteriaBuilder                         $searchCriteriaBuilder,
        GetStockItemConfigurationInterface            $getStockItemConfiguration,
        OrderItemExtensionFactory                     $orderItemExtensionFactory,
        InventoryFbOrderItemSourceRepositoryInterface $inventoryFbOrderItemSourceRepository
    )
    {
        parent::__construct($context);
        $this->isProductSalableInterface = $isProductSalableInterface;
        $this->stockResolver = $stockResolver;
        $this->websiteCollectionFactory = $websiteCollectionFactory;
        $this->productRepository = $productRepository;
        $this->stockIdResolver = $stockIdResolver;
        $this->storeManager = $storeManager;
        $this->orderRepositoryInterface = $orderRepositoryInterface;
        $this->storeRepositoryInterface = $storeRepositoryInterface;
        $this->resourceConnection = $resourceConnection;
        $this->getStockItemConfiguration = $getStockItemConfiguration;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->orderItemExtensionFactory = $orderItemExtensionFactory;
        $this->inventoryFbOrderItemSourceRepository = $inventoryFbOrderItemSourceRepository;
    }

    /**
     * @return bool
     */
    public function isEnabled(): bool
    {
        return true;
    }

    /**
     * @param $productSku
     * @param $websiteId
     * @return bool
     * @throws NoSuchEntityException
     */
    public function isSalable($productSku, $websiteId): bool
    {
        //$product = $this->getProductById($productId);
        $websiteCode = $this->getWebsiteCodeById($websiteId);
        $stockId = $this->getStockIdByWebsiteCode($websiteCode);

        if (is_object($productSku)) {
            $productSku = $productSku->getSku();
        }

        return $this->isProductSalableInterface->execute($productSku, $stockId);
    }

    /**
     * @param $websiteId
     * @return mixed
     */
    public function getWebsiteCodeById($websiteId)
    {
        $collection = $this->websiteCollectionFactory->create();
        $collection->addFieldToFilter('website_id', ['eq' => $websiteId]);
        $websiteData = $collection->getData();
        return $websiteData[0]['code'];
    }

    /**
     * @param $website_code
     * @return int|null
     * @throws NoSuchEntityException
     */
    public function getStockIdByWebsiteCode($website_code)
    {
        $stock = $this->stockResolver->execute(SalesChannelInterface::TYPE_WEBSITE, $website_code);
        return $stock->getStockId();
    }

    /**
     * @param $order
     * @return int|null
     * @throws NoSuchEntityException
     */
    public function getStockIdByOrder(OrderInterface $order)
    {
        $websiteId = $this->storeManager->getStore($order->getStoreId())->getWebsiteId();
        $websiteCode = $this->storeManager->getWebsite($websiteId)->getCode();
        $stockId = $this->stockIdResolver->resolve(SalesChannelInterface::TYPE_WEBSITE, $websiteCode);
        return $stockId;
    }

    /**
     * @return string
     * @throws NoSuchEntityException
     */
    public function getCurrentStoreviewId()
    {
        $currentStore = $this->storeManager->getStore();
        return $currentStore->getId();
    }

    /**
     * @param $id
     * @return ProductInterface|mixed
     * @throws NoSuchEntityException
     */
    public function getProductById($id)
    {
        return $this->productRepository->getById($id);
    }

    /**
     * @param $sku
     * @return ProductInterface|Product|null
     * @throws NoSuchEntityException
     */
    public function getProductBySku($sku)
    {
        return $this->productRepository->get($sku);
    }

    /**
     * @param $order_id
     * @return int|null
     */
    public function getOrderStoreviewIdByOrderId($order_id)
    {
        $order = $this->orderRepositoryInterface->get($order_id);
        return $order->getStoreId();
    }

    /**
     * @return array
     */
    public function getAllStoresIds()
    {
        $all_stores = $this->storeRepositoryInterface->getList();
        $all_stores_ids = array();
        foreach ($all_stores as $store) {
            $all_stores_ids[] = $store->getId();
        }
        return $all_stores_ids;
    }


    /**
     * @param $product
     * @param $websiteCode
     * @param null $source_code
     * @param bool $return_only_qty
     * @return mixed
     * @throws Zend_Db_Statement_Exception
     */
    public function getSalableQty($product, $websiteCode, $source_code = null, $return_only_qty = true)
    {

        if (is_string($product)) {
            $sku = [$product];
        }
        if (is_array($product)) {
            $sku = $product;
        }
        if (is_object($product)) {
            $sku = [$product->getSku()];
        }

        $connection = $this->resourceConnection->getConnection();

        // Q1 is the amount of pieces reserved by the orders
        $q1 = $connection->select()
            ->from(['isi' => $this->resourceConnection->getTableName('inventory_source_item')],
                [
                    'isi.sku',
                    "SUM(IF(so.state in('new', 'processing'),ifos.qty,0)) AS qty"
                ])
            ->joinInner(
                ['issl' => $this->resourceConnection->getTableName('inventory_source_stock_link')],
                'issl.source_code = isi.source_code', []
            )
            ->joinInner(
                ['issc' => $this->resourceConnection->getTableName('inventory_stock_sales_channel')],
                'issc.stock_id = issl.stock_id', []
            )->joinLeft(
                ['ifos' => $this->resourceConnection->getTableName('inventory_fb_orderitem_source')],
                'ifos.sku = isi.sku AND ifos.final_inventory_source_id = isi.source_code', []
            )->joinLeft(
                ['so' => $this->resourceConnection->getTableName('sales_order')],
                'ifos.order_id = so.entity_id', []
            )
            ->where('isi.sku IN (?)', $sku)
            ->where('issc.code = ?', $websiteCode)
            ->group('isi.sku');

        if ($source_code) {
            $q1->where('ifos.inventory_source_id = ?', $source_code);
        }

        // Q2 is the total availability of the products
        $q2 = $connection->select()
            ->from(['isi' => $this->resourceConnection->getTableName('inventory_source_item')],
                [
                    'isi.sku',
                    'SUM(isi.quantity) as qty'
                ])
            ->joinInner(
                ['issl' => $this->resourceConnection->getTableName('inventory_source_stock_link')],
                'issl.source_code = isi.source_code', []
            )
            ->joinInner(
                ['issc' => $this->resourceConnection->getTableName('inventory_stock_sales_channel')],
                'issc.stock_id = issl.stock_id', []
            )
            ->where('isi.sku IN (?)', $sku)
            ->where('issc.code = ?', $websiteCode)
            ->group('isi.sku');

        if ($source_code) {
            $q2->where('isi.source_code = ?', $source_code);
        }

        // Q2-Q1 is the total minus the reserved = the real availability
        $select = $connection->select()->from(
            ['q2' => new Zend_Db_Expr("($q2)")],
            ['available' => new Zend_Db_Expr('IFNULL(q2.qty,0) - IFNULL(q1.qty,0)'), 'sku' => new Zend_Db_Expr('q2.sku')]
        )->joinLeft(['q1' => new Zend_Db_Expr("($q1)")],
            'q1.sku = q2.sku', []);

        // Fetch all rows
        $productDataResult = $connection->query($select)->fetchAll();

        // Parse the result
        $result = [];
        foreach ($productDataResult as $p) {
            $result[$p['sku']] = (int)$p['available'];
        }

        // Add not found SKU -> zero quantity?
        // TODO: Throw new logic exception if the SKU passed is not found!
        foreach ($sku as $s) {
            $result[$s] = ($result[$s] ?? 0);
        }

        // Return only the quantity? Assuming one SKU is requested!
        if ($return_only_qty) {
            $reversed_array = array_reverse($result);
            return array_pop($reversed_array);
        } else {
            return $result;
        }

    }

    /**
     * @param $store
     * @return mixed|string
     */
    public function getInventoryType($store)
    {
        return $this->scopeConfig->getValue(
                self::XML_MSI_GENERAL_INVENTORY_TYPE,
                ScopeInterface::SCOPE_STORE,
                $store) ?? 'single_stock';
    }

    public function isDebugLoggingEnabled()
    {
        return $this->scopeConfig->getValue('msi/debug/logging');
    }

    /**
     * @param SourceItemInterface $sourceItem
     * @return int|float
     * @throws LocalizedException
     * @throws SkuIsNotAssignedToStockException
     */
    public function getSalableQuantityBySourceItem(SourceItemInterface $sourceItem)
    {
        /* @see \Magento\InventoryConfiguration\Model\GetLegacyStockItem::execute */
        // Stock::DEFAULT_STOCK_ID is used until we have proper multi-stock item configuration

        $stockItemConfiguration = $this->getStockItemConfiguration
            ->execute($sourceItem->getSku(), Stock::DEFAULT_STOCK_ID);

        if (!$stockItemConfiguration->isManageStock() || $stockItemConfiguration->getBackorders()) {
            // Just return a placeholder quantity when a product can can allow backorders
            return 123456789;
        }

        $connection = $this->resourceConnection->getConnection();

        if ($this->reservedQuantityBySourceSelect === null) {

            $quantityExpression = new Zend_Db_Expr('SUM(IF(soi.parent_item_id IS NULL, soi.qty_ordered - soi.qty_shipped - soi.qty_canceled - soi.qty_refunded , 0))');
            $this->reservedQuantityBySourceSelect = $connection->select()
                ->from(
                    [ 'soi' => $connection->getTableName('sales_order_item') ],
                    [ 'quantity' => $quantityExpression ])
                ->joinInner(
                    ['ifb' => $connection->getTableName('inventory_fb_orderitem_source')],
                    'soi.order_id = ifb.order_id AND ifb.sku = soi.sku',
                    []
                )
                ->where('soi.sku = :sku AND ifb.inventory_source_id = :source')
                ->group(['soi.sku']);
        }

        $quantity = $sourceItem->getQuantity() - (int)$connection->fetchOne
            (
                $this->reservedQuantityBySourceSelect,
                [
                    'sku'    => $sourceItem->getSku(),
                    'source' => $sourceItem->getSourceCode()
                ]
            );

        if ($quantity < 0) {
            return 0;
        }

        return $quantity;
    }

    /**
     * Assign FB order items because
     * on magento 2.3.0 the \Magento\Sales\Model\Order::getItems does not call the order item repository
     *
     * @param OrderInterface $order
     * @return OrderInterface
     */
    public function getFiloBluOrderItems(OrderInterface $order): OrderInterface
    {
        $orderItems = $order->getItems();

        if (empty($orderItems)) {
            return $order;
        }

        foreach ($orderItems as $orderItem) {

            /** @var OrderItemExtension $extensionAttributes */
            $extensionAttributes = $orderItem->getExtensionAttributes();

            if ($extensionAttributes === null) {
                $extensionAttributes = $this->orderItemExtensionFactory->create();
            }

            // On Magento > 2.3.0 this extension attribute it is already populated via AfterGetList::afterGetList
            /* @see \FiloBlu\ExtInventory\Plugin\Api\OrderItemRepositoryInterface\AfterGetList::afterGetList */
            if ($extensionAttributes->getFilobluOrderItem() !== null) {
                continue;
            }

            if ($orderItem->getProductType() !== Type::TYPE_SIMPLE) {
                continue;
            }

            try {
                $this->searchCriteriaBuilder->addFilter('order_item_id', $orderItem->getItemId());
                $filoBluOrderItem = $this->inventoryFbOrderItemSourceRepository
                    ->getList(
                        $this->searchCriteriaBuilder->addFilter('order_item_id', $orderItem->getItemId())
                            ->create()
                    );
                $filoBluOrderItems = $filoBluOrderItem->getItems();
                $extensionAttributes->setFilobluOrderItem($filoBluOrderItems);
                $orderItem->setExtensionAttributes($extensionAttributes);
            } catch (Exception $exception) {
                $this->_logger->critical($exception->getMessage(), ['exception' => $exception]);
                continue;
            }
        }

        return $order;
    }
}
