<?php

namespace FiloBlu\ExtInventory\Model\ShippingAlgorithms\SourceSlots;

use Exception;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\InventoryApi\Api\Data\SourceInterface;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Framework\Message\ManagerInterface;
use Magento\Inventory\Model\ResourceModel\Source\Collection as SourceCollection;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\InventoryApi\Api\GetSourcesAssignedToStockOrderedByPriorityInterface;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;
use Magento\InventoryApi\Api\GetSourceItemsBySkuInterface;
use FiloBlu\ExtInventory\Helper\Sources as HelperSources;
use FiloBlu\ExtInventory\Helper\Data as HelperData;
use Throwable;

/**
 *
 */
class SourcesData
{

    const DEFAULT_SOURCE_CODE_FALLBACK = 'magento_snatt';

    public $helperSources;
    public $helperData;
    protected $_serializer;
    protected $_messageManager;
    protected $sourcesCollection;
    protected $scopeConfig;
    protected $orderCollectionFactory;
    protected $getSourcesAssignedToStockOrderedByPriority;
    protected $getSourceItemsBySkuInterface;

    public function __construct(
        SerializerInterface                                 $serializer,
        ManagerInterface                                    $messageManager,
        SourceCollection                                    $getSourcesCollection,
        ScopeConfigInterface                                $scopeConfig,
        CollectionFactory                                   $collectionFactoryInterface,
        GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority,
        GetSourceItemsBySkuInterface                        $getSourceItemsBySkuInterface,
        HelperSources                                       $helperSources,
        HelperData                                          $helperData
    )
    {
        $this->_serializer = $serializer;
        $this->_messageManager = $messageManager;
        $this->sourcesCollection = $getSourcesCollection;
        $this->scopeConfig = $scopeConfig;
        $this->orderCollectionFactory = $collectionFactoryInterface;
        $this->getSourcesAssignedToStockOrderedByPriority = $getSourcesAssignedToStockOrderedByPriority;
        $this->getSourceItemsBySkuInterface = $getSourceItemsBySkuInterface;
        $this->helperSources = $helperSources;
        $this->helperData = $helperData;
    }

    /**
     * @return array
     * Return array of source codes and max daily slots
     * @throws NoSuchEntityException
     */
    public function getSourcesDailyMaxSlots()
    {
        // Get the daily maximum slots configuration from the backend
        $config = $this->getSourceExtraConfig();

        // Prettify the result with array key => values
        $result = [];
        foreach ($config as $key => &$config_data) {
            if ($config_data['source'] == $this->getMainSourceCode()) {
                $config_data['daily_max_slots'] = 1000000;
            }
            $result[$config_data['source']]['max'] = $config_data['daily_max_slots'];
        }
        return $result;
    }

    public function getSourceExtraConfig($skip_main_source = true, $storeview_id = null, $return_source_list_only = false)
    {
        if ($storeview_id === null) {
            $storeview_id = $this->helperData->getCurrentStoreviewId();
        }

        $config = $this->scopeConfig->getValue('msi/sources/extra_configs', ScopeInterface::SCOPE_STORE, $storeview_id);

        if (!$config) {
            $this->_messageManager->addErrorMessage('Missing system configurations for FiloBlu MSI module, they are mandatory for the selected shipping algorithm!');
            return null;
        }
        $config = $this->_serializer->unserialize($config);

        $source_list = [];
        $main_source = $this->getMainSourceCode();
        foreach ($config as $key => $data) {
            $source_list[] = $data['source'];
            if (($data['source'] == $main_source) && $skip_main_source) {
                unset($config[$key]);
            }
        }
        $source_list = array_unique($source_list);
        $config = array_values($config);

        if ($return_source_list_only) {
            return $source_list;
        }
        return $config;
    }

    /**
     * @param null $store_id
     * @return mixed|string
     * @throws NoSuchEntityException
     */
    public function getMainSourceCode($store_id = null)
    {
        $store_id = (is_numeric($store_id) ? $store_id : $this->helperData->getCurrentStoreviewId());
        $code = $this->scopeConfig->getValue('msi/sources/main_source_code', ScopeInterface::SCOPE_STORE, $store_id);
        if (!$code) {
            $code = self::DEFAULT_SOURCE_CODE_FALLBACK;
        }
        return $code;
    }

    /**
     * @param string $code
     * @param $storeview_id
     * @return mixed
     */
    public function getSourceCodeFromThirdPartiesCode($code, $storeview_id = null)
    {
        $stores_to_check = [];
        if ($storeview_id !== null) {
            $stores_to_check[] = $storeview_id;
        } else {
            $stores_to_check = $this->helperData->getAllStoresIds();
        }

        try {
            foreach ($stores_to_check as $store_id) {
                $config = $this->getSourceExtraConfig(false, $store_id);
                foreach ($config as $config_data) {
                    if ($config_data['third_parties_code'] == $code) {
                        return $config_data['source'];
                    }
                }
            }
        } catch (Throwable $e) {
        }

        return $code;
    }

    /**
     * @param $code
     * @param $storeview_id
     * @return mixed
     */
    public function getThirdPartyCodeFromSourceCode($code, $storeview_id = null)
    {
        if (!$code) {
            return null;
        }

        $stores_to_check = [];

        if ($storeview_id !== null) {
            $stores_to_check[] = $storeview_id;
        } else {
            $stores_to_check = $this->helperData->getAllStoresIds();
        }

        try {
            foreach ($stores_to_check as $store_id) {
                $config = $this->getSourceExtraConfig(false, $store_id);
                foreach ($config as $config_data) {
                    if ($config_data['source'] == $code) {
                        return $config_data['third_parties_code'];
                    }
                }
            }
        } catch (Throwable $e) {
        }
        return $code;
    }

    /**
     * @param null $store_id
     * @return mixed
     * @throws NoSuchEntityException
     */
    public function getIsGroupage($store_id = null)
    {
        $store_id = (is_numeric($store_id) ? $store_id : $this->helperData->getCurrentStoreviewId());
        return $this->scopeConfig->getValue('msi/sources/groupage', ScopeInterface::SCOPE_STORE, $store_id);
    }

    /**
     * @param null $store_id
     * @return mixed
     * @throws NoSuchEntityException
     */
    public function getOrderOccupiedSlotStatuses($store_id = null)
    {
        $store_id = (is_numeric($store_id) ? $store_id : $this->helperData->getCurrentStoreviewId());
        return $this->scopeConfig->getValue('msi/sources/occupied_slot_statuses', ScopeInterface::SCOPE_STORE, $store_id);
    }

    /**
     * @throws LocalizedException
     * @throws InputException
     */
    public function getEnabledSourcesCodesArrayOrderedByPriorityByStockId(int $stockId): array
    {
        $result = [];
        $sources = $this->getEnabledSourcesOrderedByPriorityByStockId($stockId);
        foreach ($sources as $source) {
            $result[] = $source->getSourceCode();
        }
        return $result;
    }

    /**
     * Get enabled sources ordered by priority by $stockId
     *
     * @param int $stockId
     * @return array or sourced codes (ordered by priority)
     *
     */
    public function getEnabledSourcesOrderedByPriorityByStockId(int $stockId): array
    {
        try {
            $sources = $this->getSourcesAssignedToStockOrderedByPriority->execute($stockId);
            return array_filter($sources, static function (SourceInterface $source) {
                return $source->isEnabled() && $source->getSourceCode() != 'default';
            });
        } catch (Exception $e) {
            return [];
        }

    }

    /**
     * @throws NoSuchEntityException
     */
    public function getSourcesOccupiedSlotsNumber()
    {

        $collection = $this->orderCollectionFactory->create();
        $main_source_code = $this->getMainSourceCode();

        $collection->getSelect()
            ->joinLeft(['SS' => 'sales_shipment'],
                'main_table.entity_id=SS.order_id',
                [
                    'order_entity_id' => 'main_table.entity_id',
                    'order_increment' => 'main_table.increment_id',
                    'order_status'    => 'main_table.status'
                ]
            )
            ->joinLeft(['ISS' => 'inventory_shipment_source'],
                'SS.entity_id=ISS.shipment_id',
                ['source_code' => 'ISS.source_code']
            );

        $collection->getSelect()->where("main_table.status IN ('pending','processing')");
        $collection->getSelect()->where('ISS.source_code IS NOT NULL');
        $collection->getSelect()->where("ISS.source_code <> '{$main_source_code}'");

        //$this->sourcesCollection = $this->sourcesCollection->load();
        $occupied_slots = [];
        if ((int)$collection->getSize() === 0) {
            return $occupied_slots;
        }
        // Just counting the query result = occupied slots
        foreach ($collection as $slot) {

            if (isset($occupied_slots[$slot['source_code']]['occupied'])) {
                $occupied_slots[$slot['source_code']]['occupied']++;
            } else {
                $occupied_slots[$slot['source_code']]['occupied'] = 1;
            }
        }

        return $occupied_slots;

        /*
        $sql = "SELECT main_table.entity_id AS 'order_entity_id'
                ,main_table.increment_id AS 'order_increment'
                ,main_table.status AS 'order_status'
                ,ISS.source_code
                FROM sales_order AS main_table
                LEFT JOIN sales_shipment SS ON SS.order_id = main_table.entity_id
                LEFT JOIN inventory_shipment_source ISS ON ISS.shipment_id = SS.entity_id
                WHERE main_table.status IN ('pending','processing')
                AND ISS.source_code IS NOT NULL
                AND ISS.source_code <> 'santoni_snatt'";
        */

    }


    /**
     * @param $sku
     * @return SourceItemInterface[]
     */
    public function getSourceItemsBySkuInterface($sku)
    {
        return $this->getSourceItemsBySkuInterface->execute($sku);
    }

    /**
     * @param $order_id
     * @return int|null
     */
    public function getOrderStoreviewIdByOrderId($order_id)
    {
        return $this->helperData->getOrderStoreviewIdByOrderId($order_id);
    }

}
