<?php

namespace FiloBlu\ExtInventory\Plugin\InventorySourceSelectionApi\Model\Algorithms\Result;

use FiloBlu\ExtInventory\Api\InventoryFbOrderItemSourceRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\RequestInterface;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryApi\Api\SourceItemRepositoryInterface;
use Magento\InventorySourceSelectionApi\Model\GetInStockSourceItemsBySkusAndSortedSource;
use Psr\Log\LoggerInterface;
use Magento\Framework\App\ResourceConnection;

class GetDefaultSortedSourcesResult
{
    const FIX_SOURCE_SELECTION = 'msi/plugin/fix_source_selection';
    /**
     * @var SourceItemRepositoryInterface
     */
    protected $sourceItemRepository;
    /**
     * @var SearchCriteriaBuilder
     */
    protected $searchCriteriaBuilder;
    /**
     * @var InventoryFbOrderItemSourceRepositoryInterface
     */
    protected $inventoryFbOrderItemSourceRepository;
    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;
    /**
     * @var LoggerInterface
     */
    private $logger;
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;
    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @param SourceItemRepositoryInterface $sourceItemRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param InventoryFbOrderItemSourceRepositoryInterface $inventoryFbOrderItemSourceRepository
     * @param ScopeConfigInterface $scopeConfig
     * @param LoggerInterface $logger
     * @param ResourceConnection $resourceConnection
     * @param RequestInterface $request
     */
    public function __construct(
        SourceItemRepositoryInterface $sourceItemRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        InventoryFbOrderItemSourceRepositoryInterface $inventoryFbOrderItemSourceRepository,
        ScopeConfigInterface $scopeConfig,
        LoggerInterface $logger,
        ResourceConnection $resourceConnection,
        RequestInterface $request
    )
    {
        $this->sourceItemRepository = $sourceItemRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->inventoryFbOrderItemSourceRepository = $inventoryFbOrderItemSourceRepository;
        $this->scopeConfig = $scopeConfig;
        $this->logger = $logger;
        $this->resourceConnection = $resourceConnection;
        $this->request = $request;
    }

    public function afterExecute(GetInStockSourceItemsBySkusAndSortedSource $subject, $result, array $skus, array $sortedSourceCodes)
    {
        if (!$this->scopeConfig->isSetFlag(self::FIX_SOURCE_SELECTION)) {
            return $result;
        }

        $status = [
            SourceItemInterface::STATUS_IN_STOCK
        ];

        if ($this->request->getParam('order_id')) {
            $status = [
                SourceItemInterface::STATUS_IN_STOCK,
                SourceItemInterface::STATUS_OUT_OF_STOCK
            ];
        }

        try {
            $skus = array_map('strval', $skus);
            $searchCriteria = $this->searchCriteriaBuilder
                ->addFilter(SourceItemInterface::SKU, $skus, 'in')
                ->addFilter(SourceItemInterface::SOURCE_CODE, $sortedSourceCodes, 'in')
                ->addFilter(SourceItemInterface::STATUS, $status, 'in')
                ->create();

            $items = $this->sourceItemRepository->getList($searchCriteria)->getItems();

            $itemsSorting = [];
            $itemsToUse = [];
            foreach ($items as $key => $item) {

                $fbInventoryItemsCount = $this->checkPendingOrdersSourcesBySkuAndSource($item->getSku(), $item->getSourceCode());

                if ($item->getQuantity() - $fbInventoryItemsCount <= 0) {
                    continue;
                }

                $itemsSorting[] = array_search($item->getSourceCode(), $sortedSourceCodes, true);
                $itemsToUse[$key] = $item;
            }

            if (empty($itemsToUse)) {
                return $result;
            }

            array_multisort($itemsSorting, SORT_NUMERIC, SORT_ASC, $itemsToUse);

            return $itemsToUse;
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage(), ['exception' => $e]);
        }

        return $result;
    }

    /**
     * Get pending orders with specific SKU and inventory source
     *
     * @param string $sku
     * @param string $inventorySourceId
     * @return array
     */
    public function checkPendingOrdersSourcesBySkuAndSource(string $sku, string $inventorySourceId): int
    {
        $connection = $this->resourceConnection->getConnection();

        $orderTable = $this->resourceConnection->getTableName('sales_order');
        $inventoryFbOrderItemSourceTable = $this->resourceConnection->getTableName('peuterey_mage.inventory_fb_orderitem_source');

        $states = [
            'complete',
            'closed',
            'canceled'
        ];

        $select = $connection->select()
            ->from(['x' => $inventoryFbOrderItemSourceTable])
            ->join(
                ['so' => $orderTable],
                "x.order_id = so.entity_id"
            )
            ->where('x.sku = ?', $sku)
            ->where('x.inventory_source_id = ?', $inventorySourceId)
            ->where('so.state IN (?)', $states)
            ->order('x.inventory_fb_orderitem_source_id DESC');

        return $connection->query($select)->rowCount();
    }
}
