<?php

namespace FiloBlu\ExtInventory\Model;

use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Validation\ValidationException;
use Magento\Inventory\Model\ResourceModel\SourceItem\Collection;
use Magento\Inventory\Model\ResourceModel\StockSourceLink\CollectionFactory;
use Magento\Inventory\Model\SourceItem as SourceItemModel;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryApi\Api\SourceItemsSaveInterface;
use Magento\InventoryConfigurationApi\Api\GetStockItemConfigurationInterface;
use Magento\InventoryReservationsApi\Model\GetReservationsQuantityInterface;
use Throwable;

/**
 * Class StockStatusFixer
 * @package FiloBlu\ExtInventory\Model
 */
class StockStatusFixer
{
    /**
     * @var CollectionFactory
     */
    private $sourceLinkCollectionFactory;

    /**
     * @var GetStockItemConfigurationInterface
     */
    private $getStockItemConfiguration;

    /**
     * @var \Magento\Inventory\Model\ResourceModel\SourceItem\CollectionFactory
     */
    private $sourceItemCollectionFactory;

    /**
     * @var SourceItemsSaveInterface
     */
    private $sourceItemSave;

    /**
     * @var int
     */
    private $itemsPerPage;

    /**
     * @var GetReservationsQuantityInterface
     */
    private $getReservationsQuantity;

    /**
     * StockStatusFixer constructor.
     * @param CollectionFactory $sourceLinkCollectionFactory
     * @param GetStockItemConfigurationInterface $getStockItemConfiguration
     * @param \Magento\Inventory\Model\ResourceModel\SourceItem\CollectionFactory $sourceItemCollectionFactory
     * @param GetReservationsQuantityInterface $getReservationsQuantity
     * @param SourceItemsSaveInterface $sourceItemSave
     * @param int $itemsPerPage
     */
    public function __construct(
        CollectionFactory $sourceLinkCollectionFactory,
        GetStockItemConfigurationInterface $getStockItemConfiguration,
        \Magento\Inventory\Model\ResourceModel\SourceItem\CollectionFactory $sourceItemCollectionFactory,
        GetReservationsQuantityInterface $getReservationsQuantity,
        SourceItemsSaveInterface $sourceItemSave,
        $itemsPerPage = 1000
    ) {
        $this->sourceLinkCollectionFactory = $sourceLinkCollectionFactory;
        $this->getStockItemConfiguration = $getStockItemConfiguration;
        $this->sourceItemCollectionFactory = $sourceItemCollectionFactory;
        $this->sourceItemSave = $sourceItemSave;
        $this->itemsPerPage = $itemsPerPage;
        $this->getReservationsQuantity = $getReservationsQuantity;
    }

    /**
     *
     */
    public function fixStockStatus()
    {
        $linkCollection = $this->sourceLinkCollectionFactory->create();

        foreach ($linkCollection->getItems() as $source) {
            $sourceCode = $source->getSourceCode();

            /** @var Collection $sourceItemsCollection */
            $sourceItemsCollection = $this->sourceItemCollectionFactory->create()
                ->addFilter('source_code', $sourceCode);

            $this->walkItems($sourceItemsCollection, $stockId = $source->getStockId(), [$this, 'processInventorySourceItems']);
        }
    }

    /**
     * @param Collection $sourceItemsCollection
     * @param int $stockId
     * @param callable $callback
     */
    protected function walkItems($sourceItemsCollection, $stockId, callable $callback)
    {
        $currentPage = 1;
        do {
            $items = $sourceItemsCollection->clear()->setCurPage($currentPage)->setPageSize($this->itemsPerPage)->getItems();

            $callback($stockId, $items);

            $currentPage++;
        } while ($currentPage <= $sourceItemsCollection->getLastPageNumber());
    }

    /**
     * @param int $stockId
     * @param SourceItemModel[] $sourceItems
     * @throws CouldNotSaveException
     * @throws InputException
     * @throws ValidationException
     */
    protected function processInventorySourceItems($stockId, $sourceItems = [])
    {
        $itemsToUpdate = [];

        foreach ($sourceItems as $sourceItem) {
            try {
                $sku = $sourceItem->getSku();
                $stockItemConfiguration = $this->getStockItemConfiguration->execute($sku, $stockId);

                // We don't need to Manage Stock

                if (!$stockItemConfiguration->isManageStock() || $stockItemConfiguration->getBackorders()) {
                    continue;
                }

                $quantity = $sourceItem->getQuantity();
                $reservations = $this->getReservationsQuantity->execute($sku, $stockId);
                $minimumQuantity = $stockItemConfiguration->getMinQty();
                $realStockStatus = ($quantity + $reservations - $minimumQuantity > 0) ? SourceItemInterface::STATUS_IN_STOCK : SourceItemInterface::STATUS_OUT_OF_STOCK;
            } catch (Throwable $exception) {
                echo "Deleted product {$sourceItem->getSku()} has stock ??\n";
                continue;
            }

            if ((int)$sourceItem->getData('status') !== $realStockStatus && !$stockItemConfiguration->getBackorders()) {
                echo "Wrong status for {$sourceItem->getSku()}  Updating {$sourceItem->getData('status')} => {$realStockStatus}\n";
                $sourceItem->setData('status', $realStockStatus);
                $itemsToUpdate[] = $sourceItem;
            }
        }

        if (empty($itemsToUpdate)) {
            return;
        }
        $this->sourceItemSave->execute($itemsToUpdate);
    }
}
