<?php

namespace FiloBlu\ExtInventory\Model\ShippingAlgorithms\SourceSlots;

use FiloBlu\ExtInventory\Model\ShippingAlgorithms\SourceSlots\SourcesData;
use FiloBlu\ExtInventory\Helper\Sources as HelperSources;
use Magento\Store\Model\StoreManagerInterface;
use FiloBlu\ExtInventory\Helper\Data as HelperData;

class CalculationsCore {

    const WEIGHT = '_points';

    public $order_products_quantities;
    public $current_products_sources_quantities;
    public $sources_array;
    public $sources_slots;
    public $consider_reservations = false;
    public $storeview_id = null;

    private $sourcesData;
    private $stock_id;
    public $helperSources;
    public $helperData;
    public $storeManager;

    public function __construct(
        SourcesData $sourcesData,
        HelperSources $helperSources,
        StoreManagerInterface $storeManager,
        HelperData $helperData
    )
    {
        $this->sourcesData = $sourcesData;
        $this->helperSources = $helperSources;
        $this->storeManager = $storeManager;
        $this->helperData = $helperData;
    }

    /*
     * $order_products_quantities -> array "sku -> qty"
     */
    public function calculate($order_products_quantities,$stock_id,$storeview_id = null,$consider_reservations = false){
        $this->order_products_quantities = $order_products_quantities;
        $this->stock_id = $stock_id;
        $this->consider_reservations = $consider_reservations;
        $this->storeview_id = $storeview_id;

        $all_sources = $this->helperSources->getSourcesListCodeArray($this->stock_id);
        $all_slots = $this->getSourcesSlots();

        // I need to consider only the sources and slots that I've mapped under store configuration
        $sources_enabled_for_this_order = $this->sourcesData->getSourceExtraConfig(false,$storeview_id,true);

        // Filtering the slots data
        foreach ($all_slots as $source_code => $data) {
            if (!in_array($source_code,$sources_enabled_for_this_order)){
                unset($all_slots[$source_code]);
            }
        }
        // Saving only the slots that I need to consider
        $this->sources_slots = $all_slots;

        // Filtering sources array data
        foreach ($all_sources as $key => $source_code) {
            if (!in_array($source_code,$sources_enabled_for_this_order)){
                unset($all_sources[$key]);
            }
        }
        // Saving only the sources that I need to consider
        $this->sources_array = $all_sources;

        $this->current_products_sources_quantities = $this->getCurrentProductsSourcesQuantities();

        $productsCombinationsForeachSource = $this->getProductsQuantitiesCombinationsForeachSource();
        $totalCombinations = $this->getAllValidProductsQuantitiesCombinations($productsCombinationsForeachSource);
        $orderedCombinationsByWeight = $this->calculateCombinationsWeight($totalCombinations);

        return $orderedCombinationsByWeight;
    }

    public function getCombinations ($data, &$all = array(), $group = array(), $val = null, $i = 0) {

        if ($i == 0) {
            // Converting to indexed array (only on the first call)
            $data = $this->array_values_recursive($data);
        }

        // Start calculating the combinations
        if (isset($val)) {
            array_push($group, $val);
        }
        if ($i >= count($data)) {
            array_push($all, $group);
        } else {
            foreach ($data[$i] as $v) {
                $this->getCombinations($data, $all, $group, $v, $i + 1);
            }
        }

        return $all;
    }

    // Return all the quantities combinations of an array of products
    // Array of products is equal to the order product rows (with the quantities)
    // This will return the combinations foreach store (starting from the base matrix)
    public function getProductsQuantitiesCombinationsForeachSource(){
        $base_matrix = $this->generateBaseMatrix();

        $return = array();
        foreach ($base_matrix as $source => $quantities_to_combine) {
            $combinations = $this->getCombinations($quantities_to_combine);
            // Setting back the associative array (getCombinations function works only with indexed arrays)
            foreach ($combinations as &$combination) {
                $combination = array_combine(array_keys($this->order_products_quantities),$combination);
            }
            $return[$source] = $combinations;
        }

        return $return;
    }

    public function getAllValidProductsQuantitiesCombinations($array_to_combine){
        $total_combinations = $this->getCombinations($array_to_combine);

        // Reading the real quantities considering also the reservations
        if ($this->consider_reservations){
            $real_quantities = array();
            $store = $this->storeManager->getStore($this->storeview_id);
            $website_code = $store->getWebsite()->getCode();
            $skus = array_keys($this->order_products_quantities);
            foreach ($this->sources_array as $source_code) {
                $response = $this->helperData->getSalableQty($skus,$website_code,$source_code,false);
                $real_quantities[$source_code] = $response;
            }
        }

        // Setting back the associative array for the sources (getCombinations function works only with indexed arrays)
        foreach ($total_combinations as $key => $sources_combination) {
            $sources_combination = array_combine(array_values($this->sources_array),$sources_combination);
            $total_combinations[$key] = $sources_combination;
        }

        // Setting back the associative array for the products (getCombinations function works only with indexed arrays)
        foreach ($total_combinations as $key => $sources_combination) {
            foreach ($sources_combination as $key2 => $products_combination) {
                $products_combination = array_combine(array_keys($this->order_products_quantities),$products_combination);
                $total_combinations[$key][$key2] = $products_combination;
            }
        }

        // Removing all the unvalid combinations
        // Unvalid combinations are all the combinations where the total quantity (of the combination) does not match the ordered quantity
        foreach ($total_combinations as $combination_id => $sources_combination) {
            $combinations_quantities_sum = array();
            foreach ($sources_combination as $source_code => $source_combination) {
                foreach ($source_combination as $product_sku => $quantity) {
                    // Here I'm counting the quantities of this combination
                    if(!isset($combinations_quantities_sum[$product_sku])){
                        $combinations_quantities_sum[$product_sku] = 0;
                    }
                    $combinations_quantities_sum[$product_sku] += $quantity;
                }
            }

            // Here I'm checking if the sum of the quantities of this combination is correct vs the ordered quantity
            $is_valid = true;
            foreach ($combinations_quantities_sum as $sku => $total_sum) {
                if($this->order_products_quantities[$sku] != $total_sum){
                    $is_valid = false;
                    break;
                }
            }

            if(!$is_valid){
                unset($total_combinations[$combination_id]);
                continue;
            }

            // This combination is valid, do I need to check reservations?
            if (!$this->consider_reservations){
                // No? End here
                continue;
            }

            // This solution is valid, but are all the quantities really available? Considering Reservations?
            foreach ($sources_combination as $source_code => $sku_data) {
                foreach ($sku_data as $sku => $qty_requested) {
                    if ($qty_requested == 0 || $qty_requested <= $real_quantities[$source_code][$sku]){
                        continue;
                    }
                    $is_valid = false;
                }
            }

            if(!$is_valid){
                unset($total_combinations[$combination_id]);
            }

        }

        return array_values($total_combinations);
    }

    public function getCurrentProductsSourcesQuantities(){
        $sources_quantities = array();
        foreach ($this->order_products_quantities as $sku => $quantity) {
            $sourceItems = $this->sourcesData->getSourceItemsBySkuInterface($sku);

            // Filling with zero quantity
            foreach ($this->sources_array as $source_code) {
                if (!isset($sources_quantities[$sku][$source_code])){
                    $sources_quantities[$sku][$source_code] = 0;
                }
            }

            foreach ($sourceItems as $sourceItem) {
                if($sourceItem->getSourceCode() == 'default'){
                    continue;
                }
                $sources_quantities[$sku][$sourceItem->getSourceCode()] = $sourceItem->getQuantity();
            }
        }
        return $sources_quantities;
    }

    public function getSourcesSlots(){
        $occupiedSlots = $this->sourcesData->getSourcesOccupiedSlotsNumber();
        $maxSlots = $this->sourcesData->getSourcesDailyMaxSlots();
        $slots = array_merge_recursive($occupiedSlots,$maxSlots);
        return $slots;
    }

    // FROM: ["sku123" => 3, "sku456" => 4]
    // TO: ["sku123" => [0,1,2,3], "sku456" => [0,1,2,3,4]]
    private function generateBaseMatrix(){
        $result = array();
        foreach ($this->order_products_quantities as $sku => $qty) {
            foreach ($this->sources_array as $source_code) {
                $result[$source_code][$sku] = $this->getIncrementalIntegersUntil($qty,$this->current_products_sources_quantities[$sku][$source_code]);
            }
        }
        return $result;
    }

    // Cap is the limit (not mandatory)
    private function getIncrementalIntegersUntil($number = 0,$cap = null){

        if (is_numeric($cap)){
            if ($cap < $number){
                $number = $cap;
            }
        }

        $result = array();
        foreach (range(0, $number) as $n){
            $result[] = $n;
        }
        return $result;
    }

    private function getUsedSourceWeight($how_many_used_sources = 1, $source_code = null, $number_of_products_in_this_source = 0){
        if($how_many_used_sources == 0){return 0;}

        $base_points = 1000000;
        $source_items_quantity_points = $number_of_products_in_this_source * 100;
        // Default sources, like SNATT, should weight less
        if ($source_code && $this->sourcesData->getMainSourceCode() == $source_code){
            $base_points = 900000;
            $source_items_quantity_points = $number_of_products_in_this_source * 200;
        }

        // I need to consider also how many pieces i'm withdrawing from this source (and if it is the main source, they weight more)
        $base_points = $base_points - $source_items_quantity_points;

        $number_of_skus = count($this->order_products_quantities);
        $number_of_sources = count($this->sources_array);
        $total = ($how_many_used_sources * $base_points * $number_of_skus) / $number_of_sources;
        return $total;
    }

    private function getOutOfStockWeight($how_many_out_of_stock = 1){
        if($how_many_out_of_stock == 0){
            return 0;
        }
        $base_points = 10000;
        $number_of_skus = count($this->order_products_quantities);
        $number_of_sources = count($this->sources_array);
        $total = ($how_many_out_of_stock * $base_points * $number_of_skus) / $number_of_sources;
        return $total;
    }

    private function getOccupiedSlotWeight($occupied, $max){
        if (!$occupied || (int)$max === 0){
            return 0;
        }
        $base_points = 10;
        $total = $base_points * ($occupied * 100 / $max);
        return $total;
    }

    private function getRemainingStockWeight($how_much_quantity_remains = 0){
        if($how_much_quantity_remains <= 0){return 0;}
        $base_points = 0.1;
        $number_of_skus = count($this->order_products_quantities);
        $number_of_sources = count($this->sources_array);
        $total = ($how_much_quantity_remains * $base_points * $number_of_skus) / $number_of_sources;
        return $total;
    }


    public function calculateCombinationsWeight($totalCombinations){
        foreach ($totalCombinations as $combination_id => $sources_combination) {
            // Starting with zero weight
            $totalCombinations[$combination_id][self::WEIGHT]['total'] = 0;
            foreach ($sources_combination as $source_code => $source_combination) {
                $is_source_used = false;
                foreach ($source_combination as $product_sku => $quantity) {
                    if ($quantity <= 0){
                        continue;
                    }
                    $is_source_used = true;
                    if ($quantity >= $this->current_products_sources_quantities[$product_sku][$source_code]){
                        $weight = $this->getOutOfStockWeight();
                        $totalCombinations[$combination_id][self::WEIGHT]['total'] += $weight;
                        $totalCombinations[$combination_id][self::WEIGHT]['debug']['out_of_stock'][] = $weight;
                    } else {
                        $remaining_quantity = $this->current_products_sources_quantities[$product_sku][$source_code] - $quantity;
                        $weight = $this->getRemainingStockWeight($remaining_quantity);
                        $totalCombinations[$combination_id][self::WEIGHT]['total'] -= $weight;
                        $totalCombinations[$combination_id][self::WEIGHT]['debug']['remaining_stock'][] = $weight * -1;
                    }
                }
                if($is_source_used){
                    $nr_of_products = array_sum($totalCombinations[$combination_id][$source_code]);
                    $weight = $this->getUsedSourceWeight(1,$source_code,$nr_of_products);
                    $totalCombinations[$combination_id][self::WEIGHT]['total'] += $weight;
                    $totalCombinations[$combination_id][self::WEIGHT]['debug']['used_source'][] = $weight;

                    if (isset($this->sources_slots[$source_code])){

                        if(!isset($this->sources_slots[$source_code]['occupied'])){
                            $this->sources_slots[$source_code]['occupied'] = 0;
                        }

                        $occupied = (isset($this->sources_slots[$source_code]['occupied']) ? $this->sources_slots[$source_code]['occupied'] : 0);
                        $max = (isset($this->sources_slots[$source_code]['max']) ? $this->sources_slots[$source_code]['max'] : 0);

                        $weight = $this->getOccupiedSlotWeight($occupied,$max);
                        if($weight > 0){
                            $totalCombinations[$combination_id][self::WEIGHT]['total'] += $weight;
                            $totalCombinations[$combination_id][self::WEIGHT]['debug']['slots'][] = $weight;
                        }

                    }
                }

            }
        }
        return $this->orderArrayByWeight($totalCombinations);
    }

    public function array_values_recursive($arr)
    {
        $arr2=[];
        foreach ($arr as $key => $value)
        {
            if(is_array($value))
            {
                $arr2[] = $this->array_values_recursive($value);
            }else{
                $arr2[] =  $value;
            }
        }

        return $arr2;
    }

    private function orderArrayByWeight($myArray){
        usort($myArray, function($a, $b) {
            return $a[self::WEIGHT]['total'] <=> $b[self::WEIGHT]['total'];
        });
        return $myArray;
    }

}