<?php

namespace FiloBlu\ExtInventory\Model;

use Exception;
use Magento\Framework\App\Area;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\State;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Framework\Translate\Inline\StateInterface;
use Magento\Store\Model\ScopeInterface;
use Psr\Log\LoggerInterface;

/**
 * Class ReservationEmailNotifier
 * @package FiloBlu\ExtInventory\Model
 */
class ReservationEmailNotifier
{
    /** @var string  */
    const XML_PATH_EMAIL_TEMPLATE_FIELD = 'msi/reservation/notification_template';

    /** @var string  */
    const XML_PATH_EMAIL_SENDER = 'contact/email/sender_email_identity';

    /** @var string */
    const RECEIVER_EMAILS = 'msi/reservation/notification_emails';

    /**
     * @var StateInterface
     */
    private $inlineTranslation;

    /**
     * @var TransportBuilder
     */
    private $transportBuilder;
    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;
    /**
     * @var State
     */
    private $appState;
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * ReservationEmailNotifier constructor.
     * @param StateInterface $inlineTranslation
     * @param TransportBuilder $transportBuilder
     * @param ScopeConfigInterface $scopeConfig
     * @param State $appState
     * @param LoggerInterface $logger
     */
    public function __construct(
        StateInterface $inlineTranslation,
        TransportBuilder $transportBuilder,
        ScopeConfigInterface $scopeConfig,
        State $appState,
        LoggerInterface $logger
    ) {
        $this->inlineTranslation = $inlineTranslation;
        $this->transportBuilder = $transportBuilder;
        $this->scopeConfig = $scopeConfig;
        $this->appState = $appState;
        $this->logger = $logger;
    }

    /**
     * @param \FiloBlu\ExtInventory\Model\ReservationStatusCollector $collector
     * @return $this
     * @throws \Magento\Framework\Exception\MailException
     */
    public function notify(ReservationStatusCollector $collector)
    {
        $receiversList = $this->scopeConfig->getValue(self::RECEIVER_EMAILS);
        $receiverInfo = [];
        foreach (array_map('trim', explode(',', $receiversList ?? '')) as $receiver) {
            $receiverInfo[$receiver] = ['email' => $receiver];
        }

        if (empty($receiverInfo)) {
            $this->logger->error('Unable to send reservation notification, no receiver specified');
            return $this;
        }

        /* Assign values for your template variables  */
        $variables = [
            'reservationStatusCollector' => $collector,
            'date' => (new \DateTime())->format('Y-m-d H:i:s'),
            'collector' => $collector
        ];

        $this->inlineTranslation->suspend();
        $this->generateTemplate($variables, $receiverInfo, $this->scopeConfig->getValue(self::XML_PATH_EMAIL_TEMPLATE_FIELD));

        try {
            if ($this->appState->getAreaCode() === Area::AREA_ADMINHTML) {
                $this->transportBuilder->getTransport()->sendMessage();
            } else {
                $this->appState->emulateAreaCode(Area::AREA_ADMINHTML, function () {
                    $this->transportBuilder->getTransport()->sendMessage();
                });
            }
        } catch (Exception $exception) {
            $this->appState->emulateAreaCode(Area::AREA_ADMINHTML, function () {
                $this->transportBuilder->getTransport()->sendMessage();
            });
        }

        $this->inlineTranslation->resume();

        return $this;
    }

    /**
     * @param $variable
     * @param $receiverInfo
     * @param $templateId
     * @return $this
     * @throws MailException
     */
    public function generateTemplate($variable, $receiverInfo, $templateId)
    {
        $this->transportBuilder->setTemplateIdentifier($templateId)
            ->setTemplateOptions(
                [
                    'area' => Area::AREA_ADMINHTML,
                    'store' => 0
                ]
            )
            ->setTemplateVars($variable)
            ->setFromByScope($this->emailSender());
        foreach ($receiverInfo as $receiver) {
            $this->transportBuilder->addTo($receiver['email']);
        }

        return $this;
    }

    /**
     * Return email for sender header
     * @return mixed
     */
    public function emailSender()
    {
        return $this->scopeConfig->getValue(self::XML_PATH_EMAIL_SENDER, ScopeInterface::SCOPE_STORE);
    }
}
