<?php
declare(strict_types=1);

namespace FiloBlu\ExtInventory\Model\Algorithms;

use Magento\Catalog\Model\Product\Type;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryApi\Api\SourceItemRepositoryInterface;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterfaceFactory;
use Magento\InventorySourceSelectionApi\Model\GetInStockSourceItemsBySkusAndSortedSource;
use Magento\InventorySourceSelectionApi\Model\GetSourceItemQtyAvailableInterface;
use Magento\InventorySourceSelectionApi\Model\SourceSelectionInterface;

use function array_key_exists;
use function count;

/**
 * Class FinalSourceBasedAlgorithm
 * @package FiloBlu\ExtInventory\Model\Algorithms
 */
class FinalSourceBasedAlgorithm implements SourceSelectionInterface
{
    /** @var string */
    const CONFIG_FIX_QUANTITY_SHIPMENT = 'esb/shipment/fix_quantity_shipment';

    /**
     * @var SourceSelectionResultInterfaceFactory
     */
    private $sourceSelectionResultFactory;
    /**
     * @var GetSourceItemQtyAvailableInterface
     */
    private $getSourceItemQtyAvailable;

    /**
     * @var SourceSelectionItemInterfaceFactory
     */
    private $sourceSelectionItemFactory;
    /**
     * @var SourceSelectionItemInterfaceFactory
     */
    private $scopeConfig;
    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;
    /**
     * @var SourceItemRepositoryInterface
     */
    private $sourceItemRepository;

    /**
     * FinalSourceBasedAlgorithm constructor.
     * @param SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory
     * @param GetSourceItemQtyAvailableInterface $getSourceItemQtyAvailable
     * @param GetInStockSourceItemsBySkusAndSortedSource $getInStockSourceItemsBySkusAndSortedSource
     * @param SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory
     * @param ScopeConfigInterface $scopeConfig
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SourceItemRepositoryInterface $sourceItemRepository
     * @param ResourceConnection $resourceConnection
     */
    public function __construct(
        SourceSelectionResultInterfaceFactory      $sourceSelectionResultFactory,
        GetSourceItemQtyAvailableInterface         $getSourceItemQtyAvailable,
        SourceSelectionItemInterfaceFactory        $sourceSelectionItemFactory,
        ScopeConfigInterface                       $scopeConfig,
        SearchCriteriaBuilder                      $searchCriteriaBuilder,
        SourceItemRepositoryInterface              $sourceItemRepository,
        ResourceConnection                         $resourceConnection

    )
    {
        $this->sourceSelectionResultFactory = $sourceSelectionResultFactory;
        $this->getSourceItemQtyAvailable = $getSourceItemQtyAvailable;
        $this->sourceSelectionItemFactory = $sourceSelectionItemFactory;
        $this->scopeConfig = $scopeConfig;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sourceItemRepository = $sourceItemRepository;
        $this->resourceConnection = $resourceConnection;
    }

    /**
     * @param InventoryRequestInterface $inventoryRequest
     * @return SourceSelectionResultInterface
     */
    public function execute(InventoryRequestInterface $inventoryRequest): SourceSelectionResultInterface
    {
        $extensionAttributes = $inventoryRequest->getExtensionAttributes();
        $isShipping = $extensionAttributes->getIsShipping();

        if (!$isShipping) {
            throw new \LogicException(__('FinalSourceBasedAlgorithm should be used only on shipment'));
        }

        $itemsTdDeliver = [];
        $fixSourceQuantityFlag = $this->scopeConfig->getValue(self::CONFIG_FIX_QUANTITY_SHIPMENT);
        foreach ($inventoryRequest->getItems() as $item) {
            $itemsTdDeliver[$item->getSku()] = $item->getQty();
        }

        $order = $extensionAttributes->getOrder();
        $sortedSourceCodes = [];

        foreach ($order->getItems() as $orderItem) {

            if ($orderItem->getProductType() !== Type::TYPE_SIMPLE) {
                continue;
            }

            $sku = $orderItem->getSku();
            if (!array_key_exists($sku, $itemsTdDeliver)) {
                continue;
            }
            $extensionAttributes = $orderItem->getExtensionAttributes();
            $filoBluOrderItem = $extensionAttributes->getFilobluOrderItem();
            foreach ($filoBluOrderItem as $info) {
                $sortedSourceCodes[] = $info->getFinalInventorySource()->getSourceCode();
            }
        }

        $sourceItems = $this->getSourceItems(
            array_keys($itemsTdDeliver),
            array_unique($sortedSourceCodes)
        );

        $sourceItemSelections = [];
        foreach ($sourceItems as $sourceItem) {
            $sourceItemQtyAvailable = $this->getSourceItemQtyAvailable->execute($sourceItem);
            if ($fixSourceQuantityFlag) {
                $qtyToDeduct = $itemsTdDeliver[$sourceItem->getSku()] ?? 0.0;
            } else {
                $qtyToDeduct = min($sourceItemQtyAvailable, $itemsTdDeliver[$sourceItem->getSku()] ?? 0.0);
            }

            $sourceItemSelections[] = $this->sourceSelectionItemFactory->create([
                'sourceCode'   => $sourceItem->getSourceCode(),
                'sku'          => $sourceItem->getSku(),
                'qtyToDeduct'  => $qtyToDeduct,
                'qtyAvailable' => $sourceItemQtyAvailable
            ]);
            if (!$fixSourceQuantityFlag) {
                $itemsTdDeliver[$sourceItem->getSku()] -= $qtyToDeduct;
            }
        }

        $isShippable = true;
        foreach ($itemsTdDeliver as $itemToDeliver) {
            if (!$this->isZero($itemToDeliver)) {
                // Fix Source Quantity Before Shipping Order
                if ($fixSourceQuantityFlag && count($sourceItemSelections)) {
                    $this->fixSourceQuantity($sourceItemSelections);
                } else {
                    $isShippable = false;
                    break;
                }
            }
        }

        return $this->sourceSelectionResultFactory->create(
            [
                'sourceItemSelections' => $sourceItemSelections,
                'isShippable'          => $isShippable
            ]
        );
    }

    public function getSourceItems($skus, $sortedSourceCodes)
    {
        $skus = array_map('strval', $skus);
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SourceItemInterface::SKU, $skus, 'in')
            ->addFilter(SourceItemInterface::SOURCE_CODE, $sortedSourceCodes, 'in')
            ->create();

        $items = $this->sourceItemRepository->getList($searchCriteria)->getItems();

        $itemsSorting = [];
        foreach ($items as $item) {
            $itemsSorting[] = array_search($item->getSourceCode(), $sortedSourceCodes, true);
        }

        array_multisort($itemsSorting, SORT_NUMERIC, SORT_ASC, $items);
        return $items;
    }

    /**
     * Compare float number with some epsilon
     *
     * @param float $floatNumber
     *
     * @return bool
     */
    public function isZero(float $floatNumber): bool
    {
        return $floatNumber < 0.0000001;
    }

    /**
     * @param $sourceItemSelections
     */
    protected function fixSourceQuantity($sourceItemSelections)
    {
        $connection = $this->resourceConnection->getConnection();
        $table = $connection->getTableName('inventory_source_item');

        foreach ($sourceItemSelections as $sourceItem) {

            $sourceCode = $sourceItem->getSourceCode();
            $sku = $sourceItem->getSku();

            $query = "SELECT * FROM `$table` WHERE sku  = '$sku' AND source_code = '$sourceCode'";

            $result = $connection->fetchRow($query);

            // To complete ship Magento2 need all sources to be in stock
            $update = [ 'status' => 1 ];

            if (count($result) && isset($result['quantity']) && ((int)($result['quantity']) < (int)$sourceItem->getQtyToDeduct())) {
                $update = [
                    'quantity' => $sourceItem->getQtyToDeduct(),
                    'status' => 1
                ];
            }

            $connection->update($table, $update, "sku = '$sku' AND source_code = '$sourceCode'");
        }
    }
}
