<?php

declare(strict_types=1);

namespace FiloBlu\ExtInventory\Plugin\CatalogInventory\Helper\Stock;

use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\Product\Type;
use Magento\CatalogInventory\Helper\Stock;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventorySalesApi\Api\Data\SalesChannelInterface;
use Magento\InventorySalesApi\Api\IsProductSalableInterface;
use Magento\InventorySalesApi\Api\StockResolverInterface;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Process configurable product stock status considering configurable options salable status.
 */
class AdaptAssignStatusToProductPlugin
{
    /**
     * @var Configurable
     */
    private $configurable;

    /**
     * @var IsProductSalableInterface
     */
    private $isProductSalable;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var StockResolverInterface
     */
    private $stockResolver;
    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    const IS_ENABLE_PATH = 'msi/plugin/stock_status_enable';

    /**
     * @param Configurable $configurable
     * @param IsProductSalableInterface $isProductSalable
     * @param StoreManagerInterface $storeManager
     * @param StockResolverInterface $stockResolver
     * @param ScopeConfigInterface $scopeConfig
     */
    public function __construct(
        Configurable $configurable,
        IsProductSalableInterface $isProductSalable,
        StoreManagerInterface $storeManager,
        StockResolverInterface $stockResolver,
        ScopeConfigInterface $scopeConfig
    ) {
        $this->configurable = $configurable;
        $this->isProductSalable = $isProductSalable;
        $this->storeManager = $storeManager;
        $this->stockResolver = $stockResolver;
        $this->scopeConfig = $scopeConfig;
    }

    /**
     * Process configurable product stock status, considering configurable options.
     *
     * @param Stock $subject
     * @param Product $product
     * @param int|null $status
     * @return array
     *
     * @throws LocalizedException
     * @throws NoSuchEntityException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function beforeAssignStatusToProduct(
        Stock $subject,
        Product $product,
        $status = null
    ) {
        if (!$this->scopeConfig->getValue(self::IS_ENABLE_PATH)) {
            return [$product, $status];
        }
        $website = $this->storeManager->getWebsite();
        $stock = $this->stockResolver->execute(SalesChannelInterface::TYPE_WEBSITE, $website->getCode());
        switch ($product->getTypeId()) {
            case Configurable::TYPE_CODE:

                $options = $this->configurable->getConfigurableOptions($product);
                $status = SourceItemInterface::STATUS_OUT_OF_STOCK;
                foreach ($options as $attribute) {
                    foreach ($attribute as $option) {
                        if ($this->isProductSalable->execute($option['sku'], $stock->getStockId())) {
                            $status = SourceItemInterface::STATUS_IN_STOCK;
                            return [$product, $status];
                        }
                    }
                }
                break;

            case Type::DEFAULT_TYPE:
                if ($this->isProductSalable->execute($product->getSku(), $stock->getStockId())) {
                    $status = SourceItemInterface::STATUS_IN_STOCK;
                } else {
                    $status = SourceItemInterface::STATUS_OUT_OF_STOCK;
                }
                break;
        }

        return [$product, $status];
    }
}
