<?php

namespace FiloBlu\ExtInventory\Model\ShippingAlgorithms\SourceSlots;

use Magento\Framework\App\ObjectManager;
use Magento\InventoryApi\Api\GetSourceItemsBySkuInterface;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterfaceFactory;
use Magento\InventorySourceSelectionApi\Model\Algorithms\Result\GetDefaultSortedSourcesResult;
use Magento\InventorySourceSelectionApi\Model\GetInStockSourceItemsBySkusAndSortedSource;
use Magento\InventorySourceSelectionApi\Model\GetSourceItemQtyAvailableInterface;
use Magento\InventorySourceSelectionApi\Model\SourceSelectionInterface;
use Magento\Sales\Api\Data\OrderInterface;

/**
 * Class Algorithm
 * @package FiloBlu\ExtInventory\Model\ShippingAlgorithms\SourceSlots
 */
class Algorithm implements SourceSelectionInterface
{

    //namespace Magento\InventorySourceSelectionApi\Model\Algorithms\Result;
    //class GetDefaultSortedSourcesResult
    /** @var CalculationsCore */
    protected $calculationsCore;
    /** @var */
    protected $slots;
    /** @var mixed */
    private $getDefaultSortedSourcesResult;
    /** @var SourcesData */
    private $sourcesData;
    /** @var GetSourceItemsBySkuInterface */
    private $getSourceItemsBySkuInterface;
    /** @var mixed */
    private $getInStockSourceItemsBySkusAndSortedSource;
    /** @var SourceSelectionItemInterfaceFactory */
    private $sourceSelectionItemFactory;
    /** @var GetSourceItemQtyAvailableInterface|mixed|null */
    private $getSourceItemQtyAvailable;
    /** @var SourceSelectionResultInterfaceFactory */
    private $sourceSelectionResultFactory;

    /**
     * Algorithm constructor.
     * @param SourcesData $sourcesData
     * @param GetSourceItemsBySkuInterface $getSourceItemsBySkuInterface
     * @param CalculationsCore $calculationsCore
     * @param SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory
     * @param SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory
     * @param GetDefaultSortedSourcesResult|null $getDefaultSortedSourcesResult
     * @param GetInStockSourceItemsBySkusAndSortedSource|null $getInStockSourceItemsBySkusAndSortedSource
     * @param GetSourceItemQtyAvailableInterface|null $getSourceItemQtyAvailable
     */
    public function __construct(
        SourcesData $sourcesData,
        GetSourceItemsBySkuInterface $getSourceItemsBySkuInterface,
        CalculationsCore $calculationsCore,
        SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory,
        SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory,
        GetDefaultSortedSourcesResult $getDefaultSortedSourcesResult = null,
        GetInStockSourceItemsBySkusAndSortedSource $getInStockSourceItemsBySkusAndSortedSource = null,
        GetSourceItemQtyAvailableInterface $getSourceItemQtyAvailable = null
    ) {
        $this->sourcesData = $sourcesData;
        $this->getSourceItemsBySkuInterface = $getSourceItemsBySkuInterface;
        $this->calculationsCore = $calculationsCore;
        $this->sourceSelectionResultFactory = $sourceSelectionResultFactory;
        $this->sourceSelectionItemFactory = $sourceSelectionItemFactory;
        $this->getDefaultSortedSourcesResult = $getDefaultSortedSourcesResult ?: ObjectManager::getInstance()->get(GetDefaultSortedSourcesResult::class);
        $this->getInStockSourceItemsBySkusAndSortedSource = $getInStockSourceItemsBySkusAndSortedSource ?: ObjectManager::getInstance()->get(GetInStockSourceItemsBySkusAndSortedSource::class);
        $this->getSourceItemQtyAvailable = $getSourceItemQtyAvailable ?? ObjectManager::getInstance()->get(GetSourceItemQtyAvailableInterface::class);
    }

    /**
     * @param InventoryRequestInterface $inventoryRequest
     * @return SourceSelectionResultInterface
     */
    public function execute(InventoryRequestInterface $inventoryRequest): SourceSelectionResultInterface
    {
        // Stock ID of table inventory_stock
        $stockId = $inventoryRequest->getStockId();

        $this->setSourcesSlots();

        $itemsToDeliver = [];
        foreach ($inventoryRequest->getItems() as $item) {
            $itemsToDeliver[$item->getSku()] = $item->getQty();
        }

        // Do not need to check reservations here, I'm not at the place_order event
        // (see function getResultPreviewFromOrder instead)
        $combinationsCalculated = $this->calculationsCore->calculate($itemsToDeliver, $stockId);

        return $this->buildResult($combinationsCalculated, $itemsToDeliver, $stockId);
    }

    /**
     *
     */
    private function setSourcesSlots()
    {
        // Reading the slots configuration of the sources
        $occupiedSlots = $this->sourcesData->getSourcesOccupiedSlotsNumber();
        $maxSlots = $this->sourcesData->getSourcesDailyMaxSlots();
        $this->slots = array_merge_recursive($occupiedSlots, $maxSlots);
    }

    /**
     * Result must be "SourceSelectionResultInterfaceFactory"
     * Example: Magento\InventorySourceSelectionApi\Model\Algorithms\Result\GetDefaultSortedSourcesResult
     * @param $combinationsCalculated
     * @param $itemsToDeliver
     * @param $stockId
     * @return mixed
     */
    public function buildResult($combinationsCalculated, $itemsToDeliver, $stockId)
    {
        $sortedSourceCodes = $this->sourcesData->getEnabledSourcesCodesArrayOrderedByPriorityByStockId($stockId);

        $sourceItems =
            $this->getInStockSourceItemsBySkusAndSortedSource->execute(
                array_keys($itemsToDeliver),
                $sortedSourceCodes
            );

        $sourceItemSelections = [];

        foreach ($sourceItems as $sourceItem) {
            $sourceItemQtyAvailable = $this->getSourceItemQtyAvailable->execute($sourceItem);
            $quantityToDeliverCalculated = $combinationsCalculated[0][$sourceItem->getSourceCode()][$sourceItem->getSku()];
            $qtyToDeduct = min($sourceItemQtyAvailable, $quantityToDeliverCalculated ?? 0.0);

            $sourceItemSelections[] = $this->sourceSelectionItemFactory->create([
                'sourceCode' => $sourceItem->getSourceCode(),
                'sku' => $sourceItem->getSku(),
                'qtyToDeduct' => $qtyToDeduct,
                'qtyAvailable' => $sourceItemQtyAvailable
            ]);

            $itemsToDeliver[$sourceItem->getSku()] -= $qtyToDeduct;
        }

        $isShippable = true;
        foreach ($itemsToDeliver as $itemToDeliver) {
            if (!$this->isZero($itemToDeliver)) {
                $isShippable = false;
                break;
            }
        }

        return $this->sourceSelectionResultFactory->create(
            [
                'sourceItemSelections' => $sourceItemSelections,
                'isShippable' => $isShippable
            ]
        );
    }

    /**
     * @param float $floatNumber
     * @return bool
     */
    private function isZero(float $floatNumber): bool
    {
        return $floatNumber < 0.0000001;
    }

    /**
     * @param OrderInterface $order
     * @param $stockId
     * @return mixed
     */
    public function getResultPreviewFromOrder(OrderInterface $order, $stockId)
    {
        $this->setSourcesSlots();

        $itemsToDeliver = [];
        foreach ($order->getAllVisibleItems() as $item) {
            $itemsToDeliver[$item->getSku()] = $item->getQtyOrdered();
        }

        return $this->calculationsCore->calculate($itemsToDeliver, $stockId, $order->getStoreId(),true);
    }
}
