<?php

declare(strict_types=1);

namespace FiloBlu\ExtInventory\Model;

use Exception;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Module\Manager;
use Zend_Db;

/**
 * Class ReservationCleaner
 * @package FiloBlu\ExtInventory\Model
 */
class ReservationCleaner
{
    /** @var string */
    public const CONFIG_CLEAN_DUPLICATES = 'msi/reservation/clean_duplicates';

    /** @var string */
    public const CONFIG_CLEAN_INCONSISTENT = 'msi/reservation/clean_inconsistent';

    public const CONFIG_CLEAN_INCONSISTENT_FOR_STATUS = 'msi/reservation/clean_inconsistent_for_status';

    /** @var string */
    public const CONFIG_NOTIFY_RESERVATION_ISSUE = 'msi/reservation/notify';
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;
    /**
     * @var AdapterInterface
     */
    private $connection;
    /**
     * @var string
     */
    private $reservationTable;
    /**
     * @var ReservationStatusFactory
     */
    private $reservationStatusFactory;
    /**
     * @var Manager
     */
    private $moduleManager;
    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;
    /**
     * @var ReservationStatusCollectorFactory
     */
    private $collectorFactory;
    /**
     * @var ReservationEmailNotifier
     */
    private $emailNotifier;

    /**
     * ReservationCleaner constructor.
     * @param Manager $moduleManager
     * @param ReservationStatusFactory $reservationStatusFactory
     * @param ResourceConnection $resourceConnection
     * @param ReservationStatusCollectorFactory $collectorFactory
     * @param ReservationEmailNotifier $emailNotifier
     * @param ScopeConfigInterface $scopeConfig
     */
    public function __construct(
        Manager $moduleManager,
        ReservationStatusFactory $reservationStatusFactory,
        ResourceConnection $resourceConnection,
        ReservationStatusCollectorFactory $collectorFactory,
        ReservationEmailNotifier $emailNotifier,
        ScopeConfigInterface $scopeConfig
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->reservationStatusFactory = $reservationStatusFactory;
        $this->moduleManager = $moduleManager;
        $this->scopeConfig = $scopeConfig;
        $this->collectorFactory = $collectorFactory;
        $this->emailNotifier = $emailNotifier;
    }

    /**
     * @return ReservationStatusCollector
     * @throws LocalizedException
     * @throws MailException
     * @throws NoSuchEntityException
     * @throws Exception
     */
    public function cleanAll()
    {
        $this->cleanEvents(['order_place_failed']);
        $collector = $this->collectorFactory->create();

        $orderIds = $this->getOrderIds();

        foreach ($orderIds as $orderId) {
            $collector->add($this->clean($orderId));
        }

        $this->cleanDuplicatedRows();
        $this->cleanInconsistencies();

        if ($this->scopeConfig->isSetFlag(self::CONFIG_NOTIFY_RESERVATION_ISSUE) && $collector->hasStatus(
                ReservationStatus::STATUS_ERROR
            )) {
            $this->emailNotifier->notify($collector);
        }
        return $collector;
    }

    /**
     * @param array $events
     * @return void
     */
    public function cleanEvents($events)
    {
        $connection = $this->getConnection();
        $reservationTable = $this->getReservationTable();

        $where = implode(
            ' OR ',
            array_map(static function ($event) {
                return "metadata LIKE '%\"$event\"%'";
            }, $events)
        );
        $connection->delete($reservationTable, $where);
    }

    /**
     * @return AdapterInterface
     */
    protected function getConnection(): AdapterInterface
    {
        if ($this->connection) {
            return $this->connection;
        }

        return ($this->connection = $this->resourceConnection->getConnection());
    }

    /**
     * @return string
     */
    protected function getReservationTable(): string
    {
        if ($this->reservationTable) {
            return $this->reservationTable;
        }

        return ($this->reservationTable = $this->resourceConnection->getTableName('inventory_reservation'));
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getOrderIds(): array
    {
        $ids = [];
        $connection = $this->getConnection();
        $reservationTable = $this->getReservationTable();
        $query = null;
        try {
            $query = "SELECT DISTINCT(JSON_EXTRACT(metadata , '$.object_id')) AS `order_id` FROM `{$reservationTable}`";
            $results = $connection->fetchAll($query, [], Zend_Db::FETCH_ASSOC);
        } catch (Exception $exception) {
            // check for SQL Error [1305] [42000]: FUNCTION JSON_EXTRACT does not exist
            if ((int)$exception->getCode() === 42000) {
                $query = "SELECT DISTINCT(
SUBSTRING_INDEX(MID(REPLACE(REPLACE(REPLACE(`metadata`, ' ', ''), '\t', ''), '\n', ''), 
LOCATE( '\"object_id\"', REPLACE(REPLACE(REPLACE(`metadata`, ' ', ''), '\t', ''), '\n', '') ) + 13), '\"', 1)
) AS `order_id`
FROM `{$reservationTable}`";
                $results = $connection->fetchAll($query, [], Zend_Db::FETCH_ASSOC);
            } else {
                throw $exception;
            }
        }

        foreach ($results as $result) {
            $ids[] = (int)trim($result['order_id'], " \t\n\r\0\x0B\"");
        }

        return $ids;
    }

    /**
     * As described at https://devdocs.magento.com/guides/v2.3/inventory/reservations.html
     *
     * Reservation calculations
     *
     * Magento creates a reservation for each product when the following events occur:
     * - A customer or merchant places an order.
     * - A customer or merchant fully or partially cancels an order.
     * - The merchant creates a shipment for a physical product.
     * - The merchant creates an invoice for a virtual or downloadable product.
     * - The merchant issues a credit memo.
     *
     * Reservations are append-only operations, similar to a log of events.
     * The initial reservation is assigned a negative quantity value.
     * All subsequent reservations created while processing the order are positive values.
     * When the order is complete, the sum of all reservations for the product is 0.
     *
     * @param int $orderId
     * @return ReservationStatus
     */
    public function clean($orderId): ReservationStatus
    {
        $connection = $this->getConnection();
        $reservationTable = $this->getReservationTable();

        $query = "SELECT * FROM {$reservationTable} WHERE `metadata` LIKE '%object_id\"%:%\"{$orderId}\"%'";
        $results = $connection->fetchAll($query, [], Zend_Db::FETCH_ASSOC);

        $ids = [];
        $totalQuantity = 0.0;
        /** @var ReservationStatus $reservationStatus */
        $reservationStatus = $this->reservationStatusFactory->create();
        $reservationStatus
            ->setReservations($results)
            ->setOrderId($orderId);

        foreach ($results as $result) {
            $ids[] = $result['reservation_id'];
            $totalQuantity += (float)$result['quantity'];
        }

        if ($totalQuantity === 0.0) {
            $connection->delete($reservationTable, ['reservation_id IN (?)' => $ids]);
            return $reservationStatus->setStatus(ReservationStatus::STATUS_CLEANED);
        }

        if ($totalQuantity > 0) {
            return $reservationStatus->setStatus(ReservationStatus::STATUS_ERROR);
        }

        return $reservationStatus->setStatus(ReservationStatus::STATUS_PENDING);
    }

    /**
     * Clean all duplicated rows
     */
    public function cleanDuplicatedRows(): bool
    {
        // TODO : is really needed?
        // if ($this->hasMultiStock()) {
        //  return false;
        // }

        if (!$this->scopeConfig->isSetFlag(self::CONFIG_CLEAN_DUPLICATES)) {
            return false;
        }

        $connection = $this->getConnection();
        $reservationTable = $this->getReservationTable();
        $connection->query(
            "DELETE t1 FROM $reservationTable t1
                 INNER JOIN  $reservationTable t2 
                 WHERE 
                        t1.reservation_id < t2.reservation_id AND
                        t1.sku = t2.sku AND
                        t1.stock_id = t2.stock_id AND
                        t1.quantity = t2.quantity AND
                        t1.metadata = t2.metadata"
        );

        return true;
    }

    /**
     * @return bool
     * @throws \Exception
     */
    public function cleanInconsistencies(): bool
    {
        if (!$this->scopeConfig->isSetFlag(self::CONFIG_CLEAN_INCONSISTENT)) {
            return false;
        }

        $state = $this->scopeConfig->getValue(self::CONFIG_CLEAN_INCONSISTENT_FOR_STATUS);

        if ($state === null || $state === '') {
            return false;
        }

        $orderStates = [];

        foreach (explode(',', $state) as $state) {
            if (empty($state) || ctype_space($state)) {
                continue;
            }
            $orderStates[] = $state;
        }

        if (empty($orderStates)) {
            return false;
        }

        $connection = $this->getConnection();

        $select = $connection->select()
            ->from('sales_order', 'entity_id')
            ->where('state IN (?)', $orderStates)
            ->where('entity_id IN (?)', $this->getOrderIds());

        $orderIdToClean = $connection->fetchCol($select);

        foreach ($orderIdToClean as $orderId) {
            $connection->delete('inventory_reservation', "metadata like '%\"object_id\"%:%\"{$orderId}\"%'");
        }

        return true;
    }

    /**
     * @return bool
     */
    public function hasMultiStock(): bool
    {
        if (!$this->moduleManager->isEnabled('Magento_CatalogInventory')) {
            return false;
        }

        $inventoryStock = $this->getConnection()->getTableName('inventory_stock');
        $count = (int)$this->getConnection()->fetchOne("SELECT COUNT(*) FROM $inventoryStock");

        return $count > 1;
    }
}
