<?php

namespace FiloBlu\ExtInventory\Model\Algorithms;

use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\InventoryApi\Api\GetSourcesAssignedToStockOrderedByPriorityInterface;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionItemInterfaceFactory;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterface;
use Magento\InventorySourceSelectionApi\Api\Data\SourceSelectionResultInterfaceFactory;
use Magento\InventorySourceSelectionApi\Model\GetInStockSourceItemsBySkusAndSortedSource;
use Magento\InventorySourceSelectionApi\Model\GetSourceItemQtyAvailableInterface;
use Magento\InventoryDistanceBasedSourceSelection\Model\Algorithms\DistanceBasedAlgorithm;
use Magento\InventoryApi\Api\SourceRepositoryInterface;
use FiloBlu\ExtInventory\Helper\Data as HelperData;

/**
 * Class DistanceCountryAlgorithm
 * @package FiloBlu\ExtInventory\Model\Algorithms
 */
class DistanceCountryAlgorithm extends AbstractAlgorithm
{
    /**
     * @var SourceSelectionResultInterfaceFactory
     */
    private $sourceSelectionResultFactory;
    /**
     * @var GetSourceItemQtyAvailableInterface
     */
    private $getSourceItemQtyAvailable;
    /**
     * @var GetInStockSourceItemsBySkusAndSortedSource
     */
    private $getInStockSourceItemsBySkusAndSortedSource;
    /**
     * @var SourceSelectionItemInterfaceFactory
     */
    private $sourceSelectionItemFactory;
    /**
     * @var GetSourcesAssignedToStockOrderedByPriorityInterface
     */
    private $getSourcesAssignedToStockOrderedByPriority;
    /**
     * @var HelperData
     */
    private $helperData;
    /**
     * @var distanceBasedAlgorithm
     */
    private $distanceBasedAlgorithm;
    /**
     * @var SourceRepositoryInterface
     */
    private $sourceRepository;

    /**
     * DistanceFromShippingAddressAlgorithm constructor.
     * @param SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory
     * @param GetSourceItemQtyAvailableInterface $getSourceItemQtyAvailable
     * @param GetInStockSourceItemsBySkusAndSortedSource $getInStockSourceItemsBySkusAndSortedSource
     * @param SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory
     * @param GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority
     * @param HelperData $helperData
     */
    public function __construct(
        SourceSelectionResultInterfaceFactory $sourceSelectionResultFactory,
        GetSourceItemQtyAvailableInterface $getSourceItemQtyAvailable,
        GetInStockSourceItemsBySkusAndSortedSource $getInStockSourceItemsBySkusAndSortedSource,
        SourceSelectionItemInterfaceFactory $sourceSelectionItemFactory,
        GetSourcesAssignedToStockOrderedByPriorityInterface $getSourcesAssignedToStockOrderedByPriority,
        DistanceBasedAlgorithm $distanceBasedAlgorithm,
        SourceRepositoryInterface $sourceRepository,
        HelperData $helperData
    ) {
        $this->sourceSelectionResultFactory = $sourceSelectionResultFactory;
        $this->getSourceItemQtyAvailable = $getSourceItemQtyAvailable;
        $this->getInStockSourceItemsBySkusAndSortedSource = $getInStockSourceItemsBySkusAndSortedSource;
        $this->sourceSelectionItemFactory = $sourceSelectionItemFactory;
        $this->getSourcesAssignedToStockOrderedByPriority = $getSourcesAssignedToStockOrderedByPriority;
        $this->distanceBasedAlgorithm = $distanceBasedAlgorithm;
        $this->sourceRepository = $sourceRepository;
        $this->helperData = $helperData;
    }

    /**
     * @param InventoryRequestInterface $inventoryRequest
     * @return SourceSelectionResultInterface
     * @throws InputException
     * @throws LocalizedException
     */
    public function execute(InventoryRequestInterface $inventoryRequest): SourceSelectionResultInterface
    {
        $order = $this->getOrder($inventoryRequest);

        $distanceSources = $this->distanceBasedAlgorithm->execute($inventoryRequest);

        $distanceSourcesItems = $distanceSources->getSourceSelectionItems();

        $countryId = $order->getShippingAddress()->getCountryId();
        //$orderDistanceSourcesItems = $this->getOrderSources($inventoryRequest->getStockId(), $countryId, $distanceSourcesItems);
        //$order = $this->getOrder($inventoryRequest);


        $sourcesOptions = $this->getOrderSources($inventoryRequest->getStockId(), $countryId, $inventoryRequest, $distanceSourcesItems);

        foreach ($sourcesOptions as $option) {
            if ($option['fullOrder'] && $option['country'] == $countryId) {
                return $this->sourceSelectionResultFactory->create(
                    [
                        'sourceItemSelections' => array_values($option['items']),
                        'isShippable'          => true
                    ]
                );
            }
        }

        $itemsLeft = [];
        foreach ($inventoryRequest->getItems() as $item) {
            $itemsLeft[$item->getSku()] = (float)$item->getQty();
        }

        // no full order create mixed order
        $selections = [];
        foreach ($sourcesOptions as $option) {
            /**
             * @var string $sku
             * @var SourceSelectionItemInterface $a
             */
            foreach ($option['items'] as $sku => $a) {
                if ($itemsLeft[$sku] <= 0.0) {
                    continue;
                }

                $qtyToDeduct = min($itemsLeft[$sku], $a->getQtyAvailable());

                $selections[] = $this->sourceSelectionItemFactory->create([
                    'sourceCode'   => $a->getSourceCode(),
                    'sku'          => $a->getSku(),
                    'qtyToDeduct'  => $qtyToDeduct,
                    'qtyAvailable' => $a->getQtyAvailable()
                ]);
                $itemsLeft[$sku] -= $qtyToDeduct;
            }
        }

        return $this->sourceSelectionResultFactory->create(
            [
                'sourceItemSelections' => $selections,
                'isShippable'          => true
            ]
        );
    }

    /**
     * @param $stockId
     * @param $countryId
     * @param $inventoryRequest
     * @param $sources
     * @return array
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function getOrderSources($stockId, $countryId, $inventoryRequest, $sources)
    {
        $output = [];
        $outputCountry = [];
        $sku = [];

        foreach ($inventoryRequest->getItems() as $item) {
            $sku[$item->getSku()] = (float)$item->getQty();
        }

        //$sources = $this->getSourcesAssignedToStockOrderedByPriority->execute($stockId);
        //$sources = array_reverse($sources);

        foreach ($sources as $source) {
            $itemsInSource = $this->getInStockSourceItemsBySkusAndSortedSource->execute(array_keys($sku), [$source->getSourceCode()]);

            //Fix $itemsInSource with the real quantities
            if($inventoryRequest->getExtensionAttributes()->getUseReservations()){
                $order = $this->getOrder($inventoryRequest);
                $website_code = $order->getStore()->getWebsite()->getCode();
                $realQuantities = $this->helperData->getSalableQty(array_keys($sku),$website_code,$source->getSourceCode(),false);
                foreach ($itemsInSource as $sourceItem){
                    $realQty = (isset($realQuantities[$sourceItem->getSku()]) ? (int)$realQuantities[$sourceItem->getSku()] : 0);
                    $sourceItem->setQuantity($realQty);
                }
            }

            $sourceModel = $this->sourceRepository->get($source->getSourceCode());

            $result = [
                'source'     => $source->getSourceCode(),
                'country'    => $sourceModel->getCountryId(),
                'selections' => [],
                'fullOrder'  => count($sku) === count($itemsInSource)
            ];

            foreach ($itemsInSource as $stockItem) {
                $currentSku = $stockItem->getSku();

                if ($stockItem->getQuantity() < $sku[$currentSku]) {
                    $result['items'][$currentSku] = $this->sourceSelectionItemFactory->create([
                        'sourceCode'   => $source->getSourceCode(),
                        'sku'          => $currentSku,
                        'qtyToDeduct'  => $stockItem->getQuantity(),
                        'qtyAvailable' => $stockItem->getQuantity()
                    ]);
                    $result['fullOrder'] = false;
                } else {
                    $result['items'][$currentSku] = $this->sourceSelectionItemFactory->create([
                        'sourceCode'   => $source->getSourceCode(),
                        'sku'          => $currentSku,
                        'qtyToDeduct'  => $sku[$currentSku],
                        'qtyAvailable' => $sku[$currentSku]
                    ]);
                }
            }

            if ($sourceModel->getCountryId() === $countryId) {
                array_unshift($outputCountry, $result);
                //$outputCountry[] = $result;
                //continue;
            }
            else {
                array_unshift($output, $result);
                //$output[] = $result;
            }
        }
        $output = array_merge($outputCountry, $output);
        //array_unshift($output, $outputCountry);

        return $output;
    }
}
