<?php

namespace FiloBlu\Esb\Model\Consumer;

use FiloBlu\Esb\Api\Data\MessageInterface;
use FiloBlu\Esb\Api\Data\MessageInterfaceFactory;
use FiloBlu\Esb\Api\Data\ObjectTypeDescriptorInterface;
use FiloBlu\Esb\Api\Data\StatusInterface;
use FiloBlu\Esb\Api\Data\StatusInterfaceFactory;
use FiloBlu\Esb\Converter\ConverterResolverInterface;
use FiloBlu\Esb\Core\Exception\NonRecoverableException;
use FiloBlu\Esb\Core\Expression\EvaluatorInterfaceFactory;
use FiloBlu\Esb\Core\Expression\SwitchExpressionEvaluatorFactory;
use FiloBlu\Esb\Core\Extractor\ObjectTypeFromMessage;
use FiloBlu\Esb\Framework\Template\TemplateInterfaceFactory;
use FiloBlu\Esb\Model\Saver\OrderSaver;
use Magento\Framework\DataObjectFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\HTTP\ZendClientFactory;
use Magento\Sales\Api\Data\OrderInterface;
use Psr\Log\LoggerInterface;

/**
 * Class SendOrderToEndpointConsumer
 * @package FiloBlu\Esb\Model\Consumer
 */
class SendOrderToEndpointConsumer extends SendObjectToEndpointConsumer
{
    /**
     * @var OrderSaver
     */
    private $orderSaver;

    /**
     * SendOrderToEndpointConsumer constructor.
     * @param ZendClientFactory $httpClientFactory
     * @param EvaluatorInterfaceFactory $evaluatorFactory
     * @param SwitchExpressionEvaluatorFactory $switchEvaluatorFactory
     * @param ObjectTypeDescriptorInterface $objectTypeDescriptor
     * @param MessageInterfaceFactory $messageFactory
     * @param ObjectTypeFromMessage $objectTypeFromMessage
     * @param ConverterResolverInterface $converterResolver
     * @param StatusInterfaceFactory $statusFactory
     * @param TemplateInterfaceFactory $templateFactory
     * @param DataObjectFactory $dataObjectFactory
     * @param LoggerInterface $logger
     * @param OrderSaver $orderSaver
     */
    public function __construct(
        ZendClientFactory $httpClientFactory,
        EvaluatorInterfaceFactory $evaluatorFactory,
        SwitchExpressionEvaluatorFactory $switchEvaluatorFactory,
        ObjectTypeDescriptorInterface $objectTypeDescriptor,
        MessageInterfaceFactory $messageFactory,
        ObjectTypeFromMessage $objectTypeFromMessage,
        ConverterResolverInterface $converterResolver,
        StatusInterfaceFactory $statusFactory,
        TemplateInterfaceFactory $templateFactory,
        DataObjectFactory $dataObjectFactory,
        LoggerInterface $logger,
        OrderSaver $orderSaver
    )
    {
        parent::__construct(
            $httpClientFactory,
            $evaluatorFactory,
            $switchEvaluatorFactory,
            $objectTypeDescriptor,
            $messageFactory,
            $objectTypeFromMessage,
            $converterResolver,
            $statusFactory,
            $templateFactory,
            $dataObjectFactory,
            $logger
        );
        $this->orderSaver = $orderSaver;
    }

    /**
     * @param MessageInterface $message
     * @return MessageInterface
     * @throws NoSuchEntityException
     * @throws NonRecoverableException
     */
    public function consume(MessageInterface $message): MessageInterface
    {
        $order = $this->loadObject($message);

        if (!$this->isConditionSatisfied($this->getDataForEvaluation($order))) {
            $this->getStatus()
                ->setCode(StatusInterface::SKIPPED)
                ->setOutputData('Condition was not satisfied');
            return $message;
        }

        $outputMessage = parent::consume($message);

        if ($this->getStatus()->getCode() !== StatusInterface::SUCCESS) {
            return $outputMessage;
        }

        $this->performAction($message);
        return $outputMessage;
    }

    /**
     * @inheritDoc
     */
    public function getName(): string
    {
        return 'Send Order to E.P.';
    }

    /**
     * @inheritDoc
     */
    public function getUniqueName(): string
    {
        return 'FiloBlu_Esb::send_order_to_endpoint';
    }

    /**
     * @param $message
     * @throws NonRecoverableException
     * @throws NoSuchEntityException
     */
    protected function performAction($message)
    {
        $parameters = $this->getConfiguration()->getParameters();

        $action = $parameters->getData('on_success_action');

        if ($action === null) {
            return;
        }

        /** @var OrderInterface $object */
        $object = $this->loadObject($message);
        $argument = $action['parameters'];

        switch ($action['action']) {
            case 'change_status':
                $endpoint = $parameters->getData('endpoint');
                $url = $endpoint['url'];

                // Compatibility with Magento 2.0 - 2.1
                if (method_exists($object, 'addStatusHistoryComment')) {
                    $object->addStatusHistoryComment(__('Order successfully sent to %1', $url), $argument);
                } else {
                    $object->addCommentToStatusHistory(__('Order successfully sent to %1', $url), $argument);
                }
                break;
            case 'add_comment':
                $endpoint = $parameters->getData('endpoint');
                $url = $endpoint['url'];
                // Compatibility with Magento 2.0 - 2.1
                if (method_exists($object, 'addStatusHistoryComment')) {
                    $object->addStatusHistoryComment(__('%1', $argument));
                } else {
                    $object->addCommentToStatusHistory(__('%1', $argument));
                }
                break;
            default:
                return;
        }

        $this->saveObject($object);
    }
}
