<?php

namespace FiloBlu\Esb\Model;

use FiloBlu\Esb\Api\Data\IdentifiableInterface;
use FiloBlu\Esb\Api\Data\ProducerConfigurationInterface;
use FiloBlu\Esb\Api\Data\ProducerInterface;
use FiloBlu\Esb\Api\RetryStrategyInterface;
use FiloBlu\Esb\Core\Id\UUID4GeneratorInterface;
use Magento\Framework\DataObject;
use Magento\Framework\DataObjectFactory;

/**
 * Class ProducerConfiguration
 * @package FiloBlu\Esb\Model
 */
class ProducerConfiguration implements ProducerConfigurationInterface
{
    /**
     * @var ProducerInterface
     */
    protected $producer;
    /**
     * @var int
     */
    protected $id;
    /**
     * @var string
     */
    protected $outputEvent;
    /**
     * @var DataObject
     */
    protected $parameters;
    /**
     * @var DataObjectFactory
     */
    protected $dataObjectFactory;
    /**
     * @var RetryStrategyInterface
     */
    protected $retryStrategy;
    /**
     * @var string
     */
    protected $label;
    /** @var int */

    protected $enable;
    /** @var string */
    protected $uuid;
    /**
     * @var UUID4GeneratorInterface
     */
    protected $UUID4Generator;
    /**
     * @var int
     */
    protected $priority = 1;

    /**
     * ProducerConfiguration constructor.
     * @param RetryStrategyInterface $defaultRetryStrategy
     * @param UUID4GeneratorInterface $UUID4Generator
     * @param DataObjectFactory $dataObjectFactory
     */
    public function __construct(
        RetryStrategyInterface $defaultRetryStrategy,
        UUID4GeneratorInterface $UUID4Generator,
        DataObjectFactory $dataObjectFactory
    ) {
        $this->dataObjectFactory = $dataObjectFactory;
        $this->retryStrategy = $defaultRetryStrategy;
        $this->UUID4Generator = $UUID4Generator;
    }

    /**
     * @inheritDoc
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @inheritDoc
     */
    public function setId($id): IdentifiableInterface
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getOutputEvent(): string
    {
        return $this->outputEvent;
    }

    /**
     * @inheritDoc
     */
    public function setOutputEvent($event): ProducerConfigurationInterface
    {
        $this->outputEvent = $event;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getParameters(): DataObject
    {
        return $this->parameters ?: $this->dataObjectFactory->create();
    }

    /**
     * @inheritDoc
     */
    public function setParameters(DataObject $parameters): ProducerConfigurationInterface
    {
        $this->parameters = $parameters;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getProducer(): ProducerInterface
    {
        return $this->producer->configure($this);
    }

    /**
     * @inheritDoc
     */
    public function setProducer(ProducerInterface $producer): ProducerConfigurationInterface
    {
        $this->producer = $producer;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getLabel(): string
    {
        return $this->label;
    }

    /**
     * @inheritDoc
     */
    public function setLabel(string $label): ProducerConfigurationInterface
    {
        $this->label = $label;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled(): bool
    {
        return $this->enable === self::STATUS_ENABLED;
    }

    /**
     * @inheritDoc
     */
    public function enable()
    {
        $this->enable = self::STATUS_ENABLED;
    }

    /**
     * @inheritDoc
     */
    public function disable()
    {
        $this->enable = self::STATUS_DISABLED;
    }

    /**
     * @inheritDoc
     */
    public function getRetryStrategy(): RetryStrategyInterface
    {
        return $this->retryStrategy;
    }

    /**
     * @inheritDoc
     */
    public function setRetryStrategy(RetryStrategyInterface $retryStrategy): ProducerConfigurationInterface
    {
        $this->retryStrategy = $retryStrategy;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getUuid(): string
    {
        return $this->uuid ?: $this->UUID4Generator->generate();
    }

    /**
     * @inheritDoc
     */
    public function setUuid($uuid): IdentifiableInterface
    {
        $this->uuid = $uuid;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setPriority(int $priority): ProducerConfigurationInterface
    {
        $this->priority = $priority;
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getPriority(): int
    {
        return $this->priority;
    }
}
