<?php
declare(strict_types=1);

namespace FiloBlu\Core\Plugin\Api;

use FiloBlu\Core\Api\Data\FbOrderExtensionAttrInterface;
use FiloBlu\Core\Api\FbOrderExtensionAttrRepositoryInterface;
use FiloBlu\Core\Model\FbOrderExtensionAttrFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Sales\Api\Data\OrderExtensionInterfaceFactory;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Model\Order;

class OrderRepositoryInterface
{

    /** @var FbOrderExtensionAttrRepositoryInterface */
    public $fbOrderExtensionAttrRepository;
    /** @var FbOrderExtensionAttrFactory */
    public $fbOrderExtensionAttrFactory;
    /** @var OrderExtensionInterfaceFactory */
    private $orderExtensionFactory;

    /**
     * @param FbOrderExtensionAttrFactory $fbOrderExtensionAttrFactory
     * @param FbOrderExtensionAttrRepositoryInterface $fbOrderExtensionAttrRepository
     * @param OrderExtensionInterfaceFactory $orderExtensionFactory
     */
    public function __construct(
        FbOrderExtensionAttrFactory             $fbOrderExtensionAttrFactory,
        FbOrderExtensionAttrRepositoryInterface $fbOrderExtensionAttrRepository,
        OrderExtensionInterfaceFactory          $orderExtensionFactory
    )
    {
        $this->fbOrderExtensionAttrRepository = $fbOrderExtensionAttrRepository;
        $this->fbOrderExtensionAttrFactory = $fbOrderExtensionAttrFactory;
        $this->orderExtensionFactory = $orderExtensionFactory;
    }

    /**
     * @param \Magento\Sales\Api\OrderRepositoryInterface $subject
     * @param OrderInterface | Order $order
     * @return Order|OrderInterface
     * @throws LocalizedException
     */
    public function afterSave(\Magento\Sales\Api\OrderRepositoryInterface $subject, $order)
    {

        $extensionAttributes = $order->getExtensionAttributes();
        if (!$extensionAttributes) {
            $extensionAttributes = $this->orderExtensionFactory->create();
            $order->setExtensionAttributes($extensionAttributes);
        }
        $filobluExtensionAttributes = $extensionAttributes->getFilobluExtensionAttributes();

        if (!$filobluExtensionAttributes) {
            return $order;
        }

        $filobluExtensionAttributes->setOrderId($order->getId());
        $this->fbOrderExtensionAttrRepository->save($filobluExtensionAttributes);

        return $order;
    }

    /**
     * @param \Magento\Sales\Api\OrderRepositoryInterface $subject
     * @param OrderInterface | Order $result
     * @return mixed
     * @throws LocalizedException
     */
    public function afterGetList(
        \Magento\Sales\Api\OrderRepositoryInterface $subject,
                                                    $result
    )
    {
        foreach ($result->getItems() as $order) {
            $this->afterGet($subject, $order);
        }
        return $result;
    }

    /**
     * @param \Magento\Sales\Api\OrderRepositoryInterface $subject
     * @param OrderInterface|Order $order
     * @return Order|OrderInterface
     * @throws LocalizedException
     */
    public function afterGet(
        \Magento\Sales\Api\OrderRepositoryInterface $subject,
                                                    $order
    )
    {
        if (!$order->getEntityId()) {
            return $order;
        }

        $extensionAttributes = $order->getExtensionAttributes();
        if (!$extensionAttributes) {
            $extensionAttributes = $this->orderExtensionFactory->create();
            $order->setExtensionAttributes($extensionAttributes);
        }

        try {
            $data = $this->fbOrderExtensionAttrRepository->getByOrderId($order->getId());
            $extensionAttributes->setFilobluExtensionAttributes($data);
        } catch (NoSuchEntityException $exception) {
            $extensionAttributes->setFilobluExtensionAttributes($this->fbOrderExtensionAttrFactory->create([
                'data' => [FbOrderExtensionAttrInterface::ORDER_ID => $order->getId()]
            ]));
        }


        return $order;
    }
}

