<?php

namespace FiloBlu\Core\Model;

use FiloBlu\Core\Api\StoreCalculatorInterface;
use FiloBlu\Core\Model\StoreCalculator\StoreCalculatorDefault;
use FiloBlu\Core\Api\StoreCalculatorResolverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Api\Data\StoreInterface;


/**
 *
 */
class StoreCalculatorResolver implements StoreCalculatorResolverInterface
{
    /**
     * @var array
     */
    private $resolvers;
    /**
     * @var StoreCalculatorDefault
     */
    private $defaultStrategy;

    /**
     * StoreCalculatorResolver constructor.
     * @param StoreCalculatorDefault $defaultStrategy
     * @param array $resolvers
     */
    public function __construct(
        StoreCalculatorDefault $defaultStrategy,
        array $resolvers
    ) {
        $this->resolvers = $resolvers;
        $this->defaultStrategy = $defaultStrategy;
    }

    /**
     * @inheritDoc
     */
    public function getAll()
    {
        return $this->resolvers;
    }

    /**
     * @inheritDoc
     * @throws NoSuchEntityException
     */
    public function get(string $name): StoreCalculatorInterface
    {
        $nameOrClass = trim($name, '\\');

        foreach ($this->resolvers as $key => $instance) {
            if ($key === $name) {
                return $instance;
            }

            if (get_class($instance) === $nameOrClass) {
                return $instance;
            }
        }
        throw new NoSuchEntityException(__('Could not find a store calculator with name or class "%1"', $name));
    }

    /**
     * @inheritDoc
     */
    public function getDefault(): StoreCalculatorInterface
    {
        return $this->defaultStrategy;
    }

    /**
     * @param $object
     * @return \Magento\Store\Api\Data\StoreInterface
     * @throws NoSuchEntityException
     */
    public function resolve($object) : StoreInterface
    {
        foreach ($this->resolvers as $strategy) {
            if(!$strategy->canHandle($object)){
                continue;
            }
            return $strategy->calculate($object);
        }
        throw new NoSuchEntityException(__('Could not find any store calculator that can handle the object "%1"', gettype($object)));
    }
}
