<?php
declare(strict_types=1);

namespace FiloBlu\Core\Model\Mail;

use Exception;
use Laminas\Mail\Transport\Smtp as LaminasSmtp;
use Laminas\Mail\Transport\SmtpOptions;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Mail\MessageInterface;
use Magento\Framework\Mail\TransportInterface;
use Magento\Framework\Phrase;
use Magento\Store\Model\ScopeInterface;
use Traversable;
use Zend_Mail_Transport_Smtp as ZendSmtp;

/**
 * Class TestMode
 * @package FiloBlu\Core\Model\Mail
 */
class TestMode implements TransportInterface
{

    /**
     * Configuration path to source of Return-Path and whether it should be set at all
     * @see \Magento\Config\Model\Config\Source\Yesnocustom to possible values
     */
    const XML_PATH_SENDING_SET_RETURN_PATH = 'system/smtp/set_return_path';

    /**
     * Configuration path for custom Return-Path email
     */
    const XML_PATH_SENDING_RETURN_PATH_EMAIL = 'system/smtp/return_path_email';

    /**
     * Smtp configuration
     */
    const XML_MAIL_TEST_SMTP_HOST = 'filoblu_mail/test_smtp/host';
    const XML_MAIL_TEST_SMTP_USERNAME = 'filoblu_mail/test_smtp/username';
    const XML_MAIL_TEST_SMTP_PASSWORD = 'filoblu_mail/test_smtp/password';
    const XML_MAIL_TEST_SMTP_PORT = 'filoblu_mail/test_smtp/port';
    const XML_MAIL_TEST_SMTP_SSL = 'filoblu_mail/test_smtp/ssl';
    const XML_MAIL_TEST_SMTP_CONNECTION_CLASS = 'filoblu_mail/test_smtp/connection_class';
    const XML_MAIL_TEST_SMTP_MAIL_TO = 'filoblu_mail/test_smtp/mail_to';
    const XML_MAIL_TEST_SMTP_ADD_ORIGINAL_INFO = 'filoblu_mail/test_smtp/add_original_info';

    /**
     * Whether return path should be set or no.
     *
     * Possible values are:
     * 0 - no
     * 1 - yes (set value as FROM address)
     * 2 - use custom value
     *
     * @var int
     */
    protected $isSetReturnPath;

    /**
     * @var string|null
     */
    protected $returnPathValue;

    /**
     * @var \Laminas\Mail\Transport\Smtp
     */
    private $zendTransport;

    /**
     * @var MessageInterface
     */
    private $message;
    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @param MessageInterface $message Email message object
     * @param ScopeConfigInterface $scopeConfig Core store config
     * @param null|string|array|Traversable $parameters Config options for sendmail parameters
     */
    public function __construct(
        MessageInterface $message,
        ScopeConfigInterface $scopeConfig,
        $parameters = null
    ) {
        $this->isSetReturnPath = (int)$scopeConfig->getValue(
            self::XML_PATH_SENDING_SET_RETURN_PATH,
            ScopeInterface::SCOPE_STORE
        );
        $this->returnPathValue = $scopeConfig->getValue(
            self::XML_PATH_SENDING_RETURN_PATH_EMAIL,
            ScopeInterface::SCOPE_STORE
        );

        $this->scopeConfig = $scopeConfig;
        $this->message = $message;
    }

    /**
     * @inheritdoc
     */
    public function sendMessage()
    {
        try {
            if (class_exists(LaminasSmtp ::class)) {
                $this->zendTransport  = new LaminasSmtp();
                $options = new SmtpOptions($this->getParameters());
                $this->zendTransport ->setOptions($options);
                $zendMessage = \Laminas\Mail\Message::fromString($this->message->getRawMessage())
                    ->setEncoding('utf-8');
                if (2 === $this->isSetReturnPath && $this->returnPathValue) {
                    $zendMessage->setSender($this->returnPathValue);
                } elseif (1 === $this->isSetReturnPath && $zendMessage->getFrom()->count()) {
                    $fromAddressList = $zendMessage->getFrom();
                    $fromAddressList->rewind();
                    $zendMessage->setSender($fromAddressList->current()->getEmail());
                }
            } elseif (class_exists(\Zend\Mail\Transport\Smtp::class)) {
                $smtpOptions = new \Zend\Mail\Transport\SmtpOptions($this->getParameters());
                $this->zendTransport  = new \Zend\Mail\Transport\Smtp($smtpOptions);
                $zendMessage = \Zend\Mail\Message::fromString($this->message->getRawMessage())
                    ->setEncoding('utf-8');
                if (2 === $this->isSetReturnPath && $this->returnPathValue) {
                    $zendMessage->setSender($this->returnPathValue);
                } elseif (1 === $this->isSetReturnPath && $zendMessage->getFrom()->count()) {
                    $fromAddressList = $zendMessage->getFrom();
                    $fromAddressList->rewind();
                    $zendMessage->setSender($fromAddressList->current()->getEmail());
                }
            } else {
                $zendMessage = $this->message;
                $options = $this->getParameters();
                $config = [
                    'host'     => $options['host'],
                    'port'     => $options['port'],
                    'name'     => $options['host'],
                    'auth'     => $options['connection_class'],
                    'username' => $options['connection_config']['username'],
                    'password' => $options['connection_config']['password'],
                    'ssl'      => $options['connection_config']['ssl']

                ];
                $this->zendTransport  = new ZendSmtp($options ['host'], $config);

                // TODO: To be handled
                // if (2 === $this->isSetReturnPath && $this->returnPathValue) {
                //      $zendMessage->setSender($this->returnPathValue);
                // } elseif (1 === $this->isSetReturnPath && $zendMessage->getFrom()->count()) {
                //    $fromAddressList = $zendMessage->getFrom();
                //    $fromAddressList->rewind();
                //   $zendMessage->setSender($fromAddressList->current()->getEmail());
                // }
            }

            if ($this->addOriginalInfoEnabled()) {
                $this->addOriginalInfo($zendMessage);
            }

            //Catch and replace mail to
            $zendMessage->setTo($this->getMailTo());

            //Force reply to
            $zendMessage->setReplyTo($this->getMailTo());

            //Reset CC and BCC if needed
            if (count($zendMessage->getCc())) {
                $zendMessage->setCc($this->getMailTo());
            }
            if (count($zendMessage->getBcc())) {
                $zendMessage->setTo($this->getMailTo());
            }

            //Change Subject
            $subject = '[TEST MODE] ' . $zendMessage->getSubject();
            $zendMessage->setSubject($subject);

            $this->zendTransport->send($zendMessage);
        } catch (Exception $e) {
            throw new MailException(new Phrase($e->getMessage()), $e);
        }
    }

    /**
     * @param \Laminas\Mail\Message $message
     */
    public function addOriginalInfo($message)
    {
        $headers = $message->getHeaders();
        $contentType = $headers->get('content-type')->getType();

        $originalInfos = [
            'to' => implode(',', $this->getEmailList($message->getTo())),
            'cc' => implode(',', $this->getEmailList($message->getCc())),
            'bcc' => implode(',', $this->getEmailList($message->getBcc())),
            'reply_to' => implode(',', $this->getEmailList($message->getReplyTo())),
        ];
        switch ($contentType) {
            case 'plain_text':
                $message->setBody(implode("\r\n", array_map(static function ($v, $k) {
                        return "$k : $v";
                }, $originalInfos, array_keys($originalInfos)))."\r\n".$message->getBody());
                return $message;
            case 'text/html':
                $message->setBody("<pre>Original Info\r\n\r\n".implode(
                    "\r\n",
                    array_map(static function ($v, $k) {
                            return "$k : $v";
                    }, $originalInfos, array_keys($originalInfos))
                ). '</pre><br/>' .$message->getBody());
                return $message;
            default:
                return $message;
        }
    }

    /**
     * @param $emailType
     * @return array
     */
    protected function getEmailList($emailType)
    {
        if (empty($emailType)) {
            return [];
        }
        $type = [];
        foreach ($emailType as $typeMail) {
            $type[] = $typeMail->getEmail();
        }
        return $type;
    }

    /**
     * @return array
     */
    public function getParameters(): array
    {
        $port = $this->getSmtpPort();
        $ssl = $this->getSmtpSsl();
        $username = $this->getSmtpUsername();
        $password = $this->getSmtpPassword();

        $connection_options = [
            'name' => $this->getSmtpName(),
            'host' => $this->getSmtpHost(),
            'connection_class'  => $this->getSmtpConnectionClass()
        ];

        if ($port!=null) {
            $connection_options['port'] = $port;
        }

        $connection_config = [];

        if ($username!=null) {
            $connection_config['username'] = $username;
        }
        if ($password!=null) {
            $connection_config['password'] = $password;
        }
        if ($ssl!=null) {
            $connection_config['ssl'] = $ssl;
        }

        if (count($connection_config)>0) {
            $connection_options['connection_config'] = $connection_config;
        }

        return $connection_options;
    }

    /**
     * @return mixed
     */
    public function getSmtpHost()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_TEST_SMTP_HOST);
    }

    /**
     * @return mixed
     */
    public function getSmtpName()
    {
        return $this->getSmtpHost();
    }

    /**
     * @return mixed
     */
    public function getSmtpPort()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_TEST_SMTP_PORT);
    }

    /**
     * @return mixed
     */
    public function getSmtpConnectionClass()
    {
        $connection_class = $this->scopeConfig->getValue(self::XML_MAIL_TEST_SMTP_CONNECTION_CLASS);
        return ($connection_class!=null) ? $connection_class : 'plain';
    }

    /**
     * @return mixed
     */
    public function getSmtpUsername()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_TEST_SMTP_USERNAME);
    }

    /**
     * @return mixed
     */
    public function getSmtpPassword()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_TEST_SMTP_PASSWORD);
    }

    /**
     * @return mixed
     */
    public function getSmtpSsl()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_TEST_SMTP_SSL);
    }

    /**
     * @return false|string[]
     */
    public function getMailTo()
    {
        return explode(',', $this->scopeConfig->getValue(self::XML_MAIL_TEST_SMTP_MAIL_TO));
    }

    /**
     * @return bool
     */
    public function addOriginalInfoEnabled()
    {
        return $this->scopeConfig->isSetFlag(self::XML_MAIL_TEST_SMTP_ADD_ORIGINAL_INFO);
    }

    /**
     * @inheritdoc
     */
    public function getMessage()
    {
        return $this->message;
    }
}
