<?php
declare(strict_types=1);

namespace FiloBlu\Core\Model\Mail;

use Exception;
use Laminas\Mail\Transport\Smtp as LaminasSmtp;
use Laminas\Mail\Transport\SmtpOptions;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Mail\EmailMessageInterface;
use Magento\Framework\Mail\MessageInterface;
use Magento\Framework\Mail\TransportInterface;
use Magento\Framework\Phrase;
use Magento\Store\Model\ScopeInterface;
use Psr\Log\LoggerInterface;
use Traversable;

use Zend_Mail_Transport_Smtp as ZendSmtp;

/**
 * Class Smtp
 * @package FiloBlu\Core\Model\Mail
 */
class Smtp implements TransportInterface
{

    /**
     * Configuration path to source of Return-Path and whether it should be set at all
     * @see \Magento\Config\Model\Config\Source\Yesnocustom to possible values
     */
    const XML_PATH_SENDING_SET_RETURN_PATH = 'system/smtp/set_return_path';

    /**
     * Configuration path for custom Return-Path email
     */
    const XML_PATH_SENDING_RETURN_PATH_EMAIL = 'system/smtp/return_path_email';

    /**
     * Smtp configuration
     */
    const XML_MAIL_SMTP_HOST = 'filoblu_mail/smtp/host';
    const XML_MAIL_SMTP_USERNAME = 'filoblu_mail/smtp/username';
    const XML_MAIL_SMTP_PASSWORD = 'filoblu_mail/smtp/password';
    const XML_MAIL_SMTP_PORT = 'filoblu_mail/smtp/port';
    const XML_MAIL_SMTP_SSL = 'filoblu_mail/smtp/ssl';
    const XML_MAIL_SMTP_CONNECTION_CLASS = 'filoblu_mail/smtp/connection_class';

    /**
     * Whether return path should be set or no.
     *
     * Possible values are:
     * 0 - no
     * 1 - yes (set value as FROM address)
     * 2 - use custom value
     *
     * @var int
     */
    protected $isSetReturnPath;

    /**
     * @var string|null
     */
    protected $returnPathValue;
    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;
    /**
     * @var MessageInterface
     */
    private $message;
    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;

    /**
     * @param Email $message message object
     * @param ScopeConfigInterface $scopeConfig Core store config
     * @param null|string|array|Traversable $parameters Config options for sendmail parameters
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        LoggerInterface $logger,
        $message = null,
        $parameters = null
    ) {
        $this->isSetReturnPath = (int)$scopeConfig->getValue(
            self::XML_PATH_SENDING_SET_RETURN_PATH,
            ScopeInterface::SCOPE_STORE
        );
        $this->returnPathValue = $scopeConfig->getValue(
            self::XML_PATH_SENDING_RETURN_PATH_EMAIL,
            ScopeInterface::SCOPE_STORE
        );

        $this->scopeConfig = $scopeConfig;
        $this->message = $message;
        $this->logger = $logger;
    }

    /**
     * @inheritdoc
     */
    public function sendMessage()
    {
        try {
            if (class_exists(LaminasSmtp ::class)) {
                $zendTransport = new LaminasSmtp();
                $options = new SmtpOptions($this->getParameters());
                $zendTransport->setOptions($options);
                
                $zendMessage = \Laminas\Mail\Message::fromString($this->message->getRawMessage())
                    ->setEncoding('utf-8');
                if (2 === $this->isSetReturnPath && $this->returnPathValue) {
                    $zendMessage->setSender($this->returnPathValue);
                } elseif (1 === $this->isSetReturnPath && $zendMessage->getFrom()->count()) {
                    $fromAddressList = $zendMessage->getFrom();
                    $fromAddressList->rewind();
                    $zendMessage->setSender($fromAddressList->current()->getEmail());
                }
            } elseif (class_exists(\Zend\Mail\Transport\Smtp::class)) {
                $zendTransport = new \Zend\Mail\Transport\Smtp(new \Zend\Mail\Transport\SmtpOptions($this->getParameters()));
                $zendMessage = \Zend\Mail\Message::fromString($this->message->getRawMessage())->setEncoding('utf-8');
                if (2 === $this->isSetReturnPath && $this->returnPathValue) {
                    $zendMessage->setSender($this->returnPathValue);
                } elseif (1 === $this->isSetReturnPath && $zendMessage->getFrom()->count()) {
                    $fromAddressList = $zendMessage->getFrom();
                    $fromAddressList->rewind();
                    $zendMessage->setSender($fromAddressList->current()->getEmail());
                }
            } else {
                $zendMessage = $this->message;
                $options = $this->getParameters();
                $config = [
                    'host'     => $options['host'],
                    'port'     => $options['port'],
                    'name'     => $options['host'],
                    'auth'     => $options['connection_class'],
                    'username' => $options['connection_config']['username'],
                    'password' => $options['connection_config']['password'],
                    'ssl'      => $options['connection_config']['ssl']

                ];
                $zendTransport = new ZendSmtp($options ['host'], $config);

                // TODO: To be handled
                // if (2 === $this->isSetReturnPath && $this->returnPathValue) {
                //      $zendMessage->setSender($this->returnPathValue);
                // } elseif (1 === $this->isSetReturnPath && $zendMessage->getFrom()->count()) {
                //    $fromAddressList = $zendMessage->getFrom();
                //    $fromAddressList->rewind();
                //   $zendMessage->setSender($fromAddressList->current()->getEmail());
                // }
            }

            $zendTransport->send($zendMessage);
        } catch (Exception $e) {
            $this->logger->error($e);
            throw new MailException(new Phrase($e->getMessage()), $e);
        }
    }

    /**
     * @return array
     */
    public function getParameters(): array
    {
        $port = $this->getSmtpPort();
        $ssl = $this->getSmtpSsl();
        $username = $this->getSmtpUsername();
        $password = $this->getSmtpPassword();

        $connection_options = [
            'name'             => $this->getSmtpName(),
            'host'             => $this->getSmtpHost(),
            'connection_class' => $this->getSmtpConnectionClass()
        ];

        if ($port != null) {
            $connection_options['port'] = $port;
        }

        $connection_config = [];

        if ($username != null) {
            $connection_config['username'] = $username;
        }
        if ($password != null) {
            $connection_config['password'] = $password;
        }
        if ($ssl != null) {
            $connection_config['ssl'] = $ssl;
        }

        if (count($connection_config) > 0) {
            $connection_options['connection_config'] = $connection_config;
        }

        return $connection_options;
    }

    /**
     * @return mixed
     */
    public function getSmtpPort()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_SMTP_PORT, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return mixed
     */
    public function getSmtpSsl()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_SMTP_SSL, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return mixed
     */
    public function getSmtpUsername()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_SMTP_USERNAME, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return mixed
     */
    public function getSmtpPassword()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_SMTP_PASSWORD, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return mixed
     */
    public function getSmtpName()
    {
        return $this->getSmtpHost();
    }

    /**
     * @return mixed
     */
    public function getSmtpHost()
    {
        return $this->scopeConfig->getValue(self::XML_MAIL_SMTP_HOST, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return mixed
     */
    public function getSmtpConnectionClass()
    {
        $connection_class = $this->scopeConfig->getValue(self::XML_MAIL_SMTP_CONNECTION_CLASS, ScopeInterface::SCOPE_STORE);
        return ($connection_class != null) ? $connection_class : 'plain';
    }

    /**
     * @inheritdoc
     */
    public function getMessage()
    {
        return $this->message;
    }
}
