<?php

namespace FiloBlu\Core\Model\Mail;

use Exception;
use FiloBlu\Core\Framework\Model\Mail\MailerInterface;
use FiloBlu\Core\Framework\Model\Mail\MessageInterface;
use Laminas\Mime\Part;
use Magento\Framework\App\ProductMetadataInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Mail\Template\SenderResolverInterface;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Framework\Mail\TransportInterface;
use Magento\Framework\Translate\Inline\StateInterface;
use Psr\Log\LoggerInterface;
use Throwable;

/**
 * Class Mailer
 * @package FiloBlu\Core\Model\Mail
 */
class Mailer implements MailerInterface
{
    /** @var string */
    const TYPE_OCTET_STREAM = 'application/octet-stream';
    /** @var string */
    const ENCODING_BASE64 = 'base64';
    /** @var string */
    const DISPOSITION_ATTACHMENT = 'attachment';
    /** @var StateInterface */
    private $inlineTranslation;
    /** @var TransportBuilder */
    private $transportBuilder;
    /** @var LoggerInterface */
    private $logger;
    /** @var ProductMetadataInterface */
    private $productMetadata;
    /** @var SenderResolverInterface */
    private $senderResolver;

    /**
     * Mailer constructor.
     * @param TransportBuilder $transportBuilder
     * @param StateInterface $inlineTranslation
     * @param ProductMetadataInterface $productMetadata
     * @param SenderResolverInterface $senderResolver
     * @param LoggerInterface $logger
     */
    public function __construct(
        TransportBuilder         $transportBuilder,
        StateInterface           $inlineTranslation,
        ProductMetadataInterface $productMetadata,
        SenderResolverInterface  $senderResolver,
        LoggerInterface          $logger
    ) {
        $this->transportBuilder = $transportBuilder;
        $this->inlineTranslation = $inlineTranslation;
        $this->logger = $logger;
        $this->productMetadata = $productMetadata;
        $this->senderResolver = $senderResolver;
    }

    /**
     * Send Mail
     *
     * @param MessageInterface $message
     * @return $this
     *
     * @throws LocalizedException
     * @throws MailException
     * @throws Exception
     * @throws Throwable
     */
    public function send(MessageInterface $message)
    {
        $canTranslate = $message->canTranslate();

        if (!$canTranslate) {
            $this->inlineTranslation->suspend();
        }

        $template = $message->getTemplate();

        $this->transportBuilder
            ->setTemplateIdentifier($template)
            ->setTemplateOptions($message->getTemplateOptions())
            ->setTemplateVars($message->getTemplateVariables());

        $emailFrom = $message->getFrom();

        if (is_string($emailFrom)) {
            $emailFrom = [
                'email' => $emailFrom,
                'name'  => $emailFrom
            ];
        }

        if (method_exists($this->transportBuilder, 'setFromByScope')) {
            $this->transportBuilder->setFromByScope($emailFrom);
        } else {
            $emailFrom = $this->senderResolver->resolve($emailFrom);
            $this->transportBuilder->setFrom($emailFrom);
        }

        foreach ($message->getTo() as $to) {
            $this->transportBuilder->addTo($to);
        }

        foreach ($message->getBcc() as $bcc) {
            $this->transportBuilder->addBcc($bcc);
        }

        foreach ($message->getCc() as $cc) {
            $this->transportBuilder->addCc($cc);
        }

        $replyTo = $message->getReplyTo();

        if ($replyTo) {
            $this->transportBuilder->setReplyTo($replyTo);
        }

        $transport = $this->transportBuilder->getTransport();

        $this->addAttachments($transport, $message);

        $lastException = null;

        try {
            $transport->sendMessage();
        } catch (Exception $exception) {
            $lastException = $exception;
            $this->logger->critical($exception->getMessage(), ['context' => $exception]);
        } catch (Throwable $throwable) {
            $lastException = $throwable;
            $this->logger->critical($throwable->getMessage(), ['context' => $throwable]);
        }

        if (!$canTranslate) {
            $this->inlineTranslation->resume();
        }

        if ($lastException) {
            throw $lastException;
        }

        return $this;
    }

    /**
     * @param TransportInterface $transport
     * @param MessageInterface $message
     */
    public function addAttachments(TransportInterface $transport, MessageInterface $message)
    {
        $rawAttachments = $message->getAttachments();

        if (empty($rawAttachments)) {
            return;
        }

        $zendMessage = $transport->getMessage();
        $attachments = [];

        foreach ($rawAttachments as $fileName => $attachment) {
            if (is_int($fileName)) {
                $fileName = null;
            }
            $attachments[] = $this->toAttachment($attachment, $fileName);
        }

        $parts = $zendMessage->getBody()->getParts();
        $parts = array_merge($parts, $attachments);
        if (class_exists(\Laminas\Mime\Message::class)) {
            $body = new \Laminas\Mime\Message();
        } else {
            $body = new \Zend\Mime\Message();
        }
        $body->setParts($parts);
        $zendMessage->setBody($body);
    }

    /**
     * Add an attachment to the message.
     *
     * @param string $file
     * @param string $fileName
     */
    public function toAttachment($file, $fileName = null)
    {
        $mime = self::TYPE_OCTET_STREAM;

        if (is_file($file)) {
            $data = file_get_contents($file);

            if (function_exists('finfo_open')) {
                $finfo = finfo_open(FILEINFO_MIME);
                $mime = finfo_buffer($finfo, $data, FILEINFO_MIME);
                finfo_close($finfo);
            } elseif (function_exists('mime_content_type')) {
                $mime = mime_content_type($file);
            }
        } else {
            $data = $file;
        }

        if (class_exists(Part::class)) {
            $part = new  Part();
        } else {
            $part = new \Zend\Mime\Part();
        }

        if (!$fileName) {
            $fileName = pathinfo($file, PATHINFO_FILENAME) . '.' . pathinfo($file, PATHINFO_EXTENSION);
        }

        $part->setType($mime)
            ->setFileName($fileName)
            ->setDisposition(self::DISPOSITION_ATTACHMENT)
            ->setEncoding(self::ENCODING_BASE64)
            ->setContent($data);
        return $part;
    }

    /**
     * @return bool
     */
    protected function isBelow233()
    {
        return version_compare($this->productMetadata->getVersion(), '2.3.3', '<');
    }
}
