<?php

declare(strict_types=1);

namespace FiloBlu\Core\Model\Mail;

use Exception;
use FiloBlu\Core\Framework\Model\Serializer;
use Laminas\Mail\Transport\Smtp as LaminasSmtp;
use Laminas\Mail\Transport\SmtpOptions;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Mail\MessageInterface;
use Magento\Framework\Mail\TransportInterface;
use Magento\Framework\Phrase;
use Zend_Mail_Transport_Smtp as ZendSmtp;

/**
 * Class DynamicSmtp
 * @package FiloBlu\Core\Model\Mail
 */
class DynamicSmtp implements TransportInterface
{

    /**
     * Configuration path to source of Return-Path and whether it should be set at all
     * @see \Magento\Config\Model\Config\Source\Yesnocustom to possible values
     */
    const XML_PATH_SENDING_SET_RETURN_PATH = 'system/smtp/set_return_path';

    /**
     * Configuration path for custom Return-Path email
     */
    const XML_PATH_SENDING_RETURN_PATH_EMAIL = 'system/smtp/return_path_email';

    /** @var string */
    const XML_PATH_FILOBLU_MAIL_DYNAMIC_SMTP_SEND_GRID = 'filoblu_mail/dynamic_smtp/send_grid';


    /**
     * Whether return path should be set or no.
     *
     * Possible values are:
     * 0 - no
     * 1 - yes (set value as FROM address)
     * 2 - use custom value
     *
     * @var int
     */
    protected $isSetReturnPath;

    /**
     * @var string|null
     */
    protected $returnPathValue;
    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;
    /**
     * @var array|null
     */
    protected $sendGrid;
    /**
     * @var LaminasSmtp
     */
    private $zendTransport;
    /**
     * @var MessageInterface
     */
    private $message;
    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * DynamicSmtp constructor.
     * @param MessageInterface $message
     * @param ScopeConfigInterface $scopeConfig
     * @param Serializer $serializer
     * @param null $parameters
     */
    public function __construct(
        MessageInterface     $message,
        ScopeConfigInterface $scopeConfig,
        Serializer           $serializer,
        $parameters = null
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->message = $message;
        $this->serializer = $serializer;
    }

    /**
     * @inheritdoc
     */
    public function sendMessage()
    {
        try {
            if (class_exists(LaminasSmtp ::class)) {
                $this->zendTransport = new LaminasSmtp();
                $zendMessage = \Laminas\Mail\Message::fromString($this->message->getRawMessage())->setEncoding('utf-8');
                $from = $zendMessage->getFrom()->current()->getEmail();
                $options = new SmtpOptions($this->getParameteres($from));
                $this->zendTransport->setOptions($options);
            } elseif (class_exists(\Zend\Mail\Transport\Smtp::class)) {
                $zendMessage = \Zend\Mail\Message::fromString($this->message->getRawMessage())->setEncoding('utf-8');
                $from = $zendMessage->getFrom()->current()->getEmail();
                $this->zendTransport = new \Zend\Mail\Transport\Smtp(new \Zend\Mail\Transport\SmtpOptions($this->getParameteres($from)));
            } else {
                $zendMessage = $this->message;
                $from = $zendMessage->getFrom();
                $options = $this->getParameteres($from);
                $config = [
                    'host'     => $options['host'],
                    'port'     => $options['port'],
                    'name'     => $options['host'],
                    'auth'     => $options['connection_class'],
                    'username' => $options['connection_config']['username'],
                    'password' => $options['connection_config']['password'],
                    'ssl'      => $options['connection_config']['ssl']

                ];
                $this->zendTransport = new ZendSmtp($options ['host'], $config);
            }


            // TODO: must be handled?
            //  $this->isSetReturnPath = (int)$this->scopeConfig->getValue(
            //    self::XML_PATH_SENDING_SET_RETURN_PATH,
            //   ScopeInterface::SCOPE_STORE );
            //    $this->returnPathValue = $this->scopeConfig->getValue(
            //      self::XML_PATH_SENDING_RETURN_PATH_EMAIL,
            //      ScopeInterface::SCOPE_STORE
            //  );
            //   if (2 === $this->isSetReturnPath && $this->returnPathValue) {
            //      $zendMessage->setSender($this->returnPathValue);
            //    } elseif (1 === $this->isSetReturnPath && $zendMessage->getFrom()->count()) {
            //        $fromAddressList = $zendMessage->getFrom();
            //       $fromAddressList->rewind();
            //      $zendMessage->setSender($fromAddressList->current()->getEmail());
            //   }

            $this->zendTransport->send($zendMessage);
        } catch (Exception $e) {
            throw new MailException(new Phrase($e->getMessage()), $e);
        }
    }

    /**
     * @param string $email
     * @return array
     */
    public function getParameteres(string $email): array
    {
        $configuration = $this->getConfiguration($email);
        return [
            'host'              => $configuration['host'],
            'port'              => $configuration['port'],
            'name'              => $configuration['host'],
            'connection_class'  => $configuration['login'],
            'connection_config' => [
                'username' => $configuration['username'],
                'password' => $configuration['password'],
                'ssl'      => $configuration['ssl']
            ]
        ];
    }

    /**
     * @param $email
     * @return mixed
     */
    public function getConfiguration($email)
    {
        $default = null;

        $data = $this->serializer->unserialize($this->scopeConfig->getValue(self::XML_PATH_FILOBLU_MAIL_DYNAMIC_SMTP_SEND_GRID));

        foreach ($data as $data) {
            if ($default === null) {
                $default = $data;
            }

            if ($data['from'] === $email) {
                return $data;
            }
        }

        return $default;
    }

    /**
     * @inheritdoc
     */
    public function getMessage()
    {
        return $this->message;
    }
}
