<?php
declare(strict_types=1);

namespace FiloBlu\Core\Model;

use FiloBlu\Core\Api\Data\FbOrderExtensionAttrInterface;
use FiloBlu\Core\Api\Data\FbOrderExtensionAttrInterfaceFactory;
use FiloBlu\Core\Api\FbOrderExtensionAttrRepositoryInterface;
use FiloBlu\Core\Framework\Model\AbstractSqlRepository;
use FiloBlu\Core\Framework\Model\RepositoryItem;
use Magento\Framework\Api\Search\SearchCriteriaInterfaceFactory;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\NoSuchEntityException;
use Zend_Db;

/**
 *
 */
class FbOrderExtensionAttrRepository extends AbstractSqlRepository implements FbOrderExtensionAttrRepositoryInterface
{

    /**
     *
     */
    const TABLE_NAME = 'fb_extensionattributes_order';

    /** @var FbOrderExtensionAttrInterfaceFactory */
    protected $dataFbOrderExtensionAttrFactory;

    /**
     * AbstractSqlRepository constructor.
     * @param ResourceConnection $resourceConnection
     * @param SearchCriteriaInterfaceFactory $searchCriteriaFactory
     * @param SearchResultsInterfaceFactory $searchResultsFactory
     * @param FbOrderExtensionAttrInterfaceFactory $dataFbOrderExtensionAttrFactory
     */
    public function __construct(
        ResourceConnection                   $resourceConnection,
        SearchCriteriaInterfaceFactory       $searchCriteriaFactory,
        SearchResultsInterfaceFactory        $searchResultsFactory,
        FbOrderExtensionAttrInterfaceFactory $dataFbOrderExtensionAttrFactory
    )
    {
        parent::__construct(
            $resourceConnection,
            $searchCriteriaFactory,
            $searchResultsFactory
        );
        $this->dataFbOrderExtensionAttrFactory = $dataFbOrderExtensionAttrFactory;
    }

    /**
     * {@inheritdoc}
     */
    public function save(
        FbOrderExtensionAttrInterface $fbOrderExtensionAttr
    )
    {
        $connection = $this->resourceConnection->getConnection();
        $table = $connection->getTableName(self::TABLE_NAME);

        $bindings = [
            FbOrderExtensionAttrInterface::ORDER_ID => $fbOrderExtensionAttr->getOrderId(),
            FbOrderExtensionAttrInterface::PREPRINTED_RMA_TRACKING_CODE => $fbOrderExtensionAttr->getPreprintedRmaTrackingCode(),
            FbOrderExtensionAttrInterface::SHIPPING_DOCUMENT_PAYLOAD => $fbOrderExtensionAttr->getShippingDocumentPayload(),
            FbOrderExtensionAttrInterface::SHIPPING_DOCUMENT_URL => $fbOrderExtensionAttr->getShippingDocumentUrl()
        ];

        $connection->insertOnDuplicate(
            $table,
            $bindings,
            [
                FbOrderExtensionAttrInterface::ORDER_ID,
                FbOrderExtensionAttrInterface::PREPRINTED_RMA_TRACKING_CODE,
                FbOrderExtensionAttrInterface::SHIPPING_DOCUMENT_PAYLOAD,
                FbOrderExtensionAttrInterface::SHIPPING_DOCUMENT_URL
            ]);

        $fbOrderExtensionAttr->setId($connection->fetchOne($connection->select()
            ->from(
                $table,
                [FbOrderExtensionAttrInterface::FBORDEREXTENSIONATTR_ID => FbOrderExtensionAttrInterface::FBORDEREXTENSIONATTR_ID])
            ->where(sprintf('%s = ?', FbOrderExtensionAttrInterface::ORDER_ID), $fbOrderExtensionAttr->getOrderId())
        ));

        return $fbOrderExtensionAttr;
    }

    /**
     * @return string
     */
    public function getIdFieldName(): string
    {
        return FbOrderExtensionAttrInterface::FBORDEREXTENSIONATTR_ID;
    }

    /**
     * @param $fborderextensionattrId
     * @return FbOrderExtensionAttrInterface|RepositoryItem
     * @throws NoSuchEntityException
     */
    public function get($fborderextensionattrId)
    {
        return $this->getById($fborderextensionattrId);
    }

    /**
     * @param $orderId
     * @return FbOrderExtensionAttrInterface|mixed
     * @throws NoSuchEntityException
     */
    public function getByOrderId($orderId)
    {
        $connection = $this->resourceConnection->getConnection();
        $table = $connection->getTableName($this->getTable());

        $select = $connection->select()->from($table)->where(sprintf('%s = :%s', FbOrderExtensionAttrInterface::ORDER_ID, FbOrderExtensionAttrInterface::ORDER_ID));

        $item = $connection->fetchRow($select, [FbOrderExtensionAttrInterface::ORDER_ID => $orderId], Zend_Db::FETCH_ASSOC);

        if (empty($item)) {
            throw new NoSuchEntityException(__("Could not find item with id = $orderId"));
        }

        return $this->fromRow($item);
    }

    /**
     * @return string
     */
    public function getTable(): string
    {
        return self::TABLE_NAME;
    }

    /**
     * @param array $row
     * @return FbOrderExtensionAttrInterface|mixed
     */
    public function fromRow(array $row)
    {
        return $this->dataFbOrderExtensionAttrFactory->create(
            [
                'data' => $row
            ]
        );
    }

    /**
     * @param FbOrderExtensionAttrInterface $fbOrderExtensionAttr
     * @return bool|void
     */
    public function delete(FbOrderExtensionAttrInterface $fbOrderExtensionAttr)
    {
        $this->deleteById($fbOrderExtensionAttr->getId());
    }
}

