<?php

declare(strict_types=1);

namespace FiloBlu\Core\Model;

use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Api\Data\WebsiteInterface;
use Magento\Store\Model\Store;
use RuntimeException;

/**
 *
 */
class DomainGroup
{

    private $storesByCode = [];

    private $storesById = [];


    /**
     * @var string
     */
    private $name;
    /**
     * @var string
     */
    private $projectId;

    /**
     * @var StoreInterface
     */
    private $defaultStore;
    /**
     * @var  WebsiteInterface[]
     */
    private $websitesById = [];
    /**
     * @var  WebsiteInterface[]
     */
    private $websitesByCode = [];

    /**
     * @return string|null
     */
    public function getProjectId()
    {
        return $this->projectId;
    }

    /**
     * @param string $projectId
     * @return $this
     */
    public function setProjectId(string $projectId): DomainGroup
    {
        $this->projectId = $projectId;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function setName(string $name): DomainGroup
    {
        $this->name = $name;
        return $this;
    }

    /**
     * @param Store|StoreInterface $store
     * @return DomainGroup
     * @throws NoSuchEntityException
     */
    public function addStore(StoreInterface $store): DomainGroup
    {
        $this->storesByCode[$store->getCode()] = $store;
        $this->storesById[$store->getId()] = $store;
        $this->websitesById[$store->getWebsite()->getId()] = $store->getWebsite();
        $this->websitesByCode[$store->getWebsite()->getCode()] = $store->getWebsite();

        return $this;
    }

    public function getDefaultStore(): StoreInterface
    {
        return $this->defaultStore;
    }

    public function setDefaultStore(StoreInterface $store): DomainGroup
    {
        $this->defaultStore = $store;
        return $this;
    }

    /**
     * @param WebsiteInterface|StoreInterface $entity
     * @return bool
     */
    public function has($entity): bool
    {

        switch (true) {
            case $entity instanceof WebsiteInterface:
                return array_key_exists($entity->getId(), $this->websitesById);
            case $entity instanceof StoreInterface:
                return array_key_exists($entity->getId(), $this->storesById);
            default:
                throw new RuntimeException('Only WebsiteInterface and StoreInterface as supported');
        }
    }

    /**
     * @return StoreInterface[]
     */
    public function getStoresById(): array
    {
        return $this->storesById;
    }

    /**
     * @return StoreInterface[]
     */
    public function getStoresByCode(): array
    {
        return $this->storesByCode;
    }

    /**
     * @return string[]
     */
    public function getStoreIds(): array
    {
        return array_keys($this->storesById);
    }

    /**
     * @return string[]
     */
    public function getStoreCodes(): array
    {
        return array_keys($this->storesByCode);
    }

    /**
     * @return WebsiteInterface[]
     */
    public function getWebsitesById(): array
    {
        return $this->websitesById;
    }

    /**
     * @return WebsiteInterface[]
     */
    public function getWebsitesByCode(): array
    {
        return $this->websitesByCode;
    }

    /**
     * @return string[]
     */
    public function getWebsiteIds(): array
    {
        return array_keys($this->websitesById);
    }

    /**
     * @return string[]
     */
    public function getWebsiteCodes(): array
    {
        return array_keys($this->websitesByCode);
    }
}
