<?php
declare(strict_types=1);

namespace FiloBlu\Core\Model;

use FiloBlu\Core\Model\Serializer\SerializerInterface;
use InvalidArgumentException;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Api\Data\WebsiteInterface;
use Magento\Store\Model\StoreManagerInterface;
use RuntimeException;

/**
 * Class Configuration
 * @package FiloBlu\Core\Helper
 */
class Configuration
{

    const XML_PATH_FILO_GENERAL_MAPPING = 'filoblu/general/mapping';

    const XML_PATH_FILO_GENERAL_MULTIDOMAIN = 'filoblu/general/multidomain';

    const XML_PATH_FILO_GENERAL_PROJECT_ID = 'filoblu/general/project_id';


    /** @var ScopeConfigInterface */
    private $scopeConfig;

    /** @var SerializerInterface */
    private $serializer;

    /** @var array|null */
    private $mapping;

    /** @var StoreManagerInterface */
    private $storeManager;

    /** @var DomainGroupFactory */
    private $domainGroupFactory;

    /**
     * Configuration constructor.
     * @param ScopeConfigInterface $scopeConfig
     * @param SerializerInterface $serializer
     * @param StoreManagerInterface $storeManager
     * @param DomainGroupFactory $domainGroupFactory
     */
    public function __construct(
        ScopeConfigInterface  $scopeConfig,
        SerializerInterface   $serializer,
        StoreManagerInterface $storeManager,
        DomainGroupFactory    $domainGroupFactory
    )
    {
        $this->scopeConfig = $scopeConfig;
        $this->serializer = $serializer;
        $this->storeManager = $storeManager;
        $this->domainGroupFactory = $domainGroupFactory;
    }

    /**
     * @param WebsiteInterface|StoreInterface $scope
     * @return string
     * @throws LocalizedException
     */
    public function getProjectId($scope = null): string
    {
        foreach ($this->getWebsiteMapping() as $map) {
            if ($map->has($scope)) {
                return $map->getProjectId();
            }
        }

        throw new RuntimeException('No project id available');
    }

    /**
     * @return DomainGroup[]
     * @throws LocalizedException
     */
    public function getWebsiteMapping()
    {
        if (!$this->mapping) {
            $mapping = [];
            $this->mapping = [];
            try {
                $mapping = $this->serializer
                    ->unserialize($this->scopeConfig->getValue(self::XML_PATH_FILO_GENERAL_MAPPING));
            } catch (InvalidArgumentException $exception) {

            }

            if (!$this->isMultiDomain()) {
                $mapping = [
                    [
                        'name'          => 'Default',
                        'project_id'    => $this->scopeConfig->getValue(self::XML_PATH_FILO_GENERAL_PROJECT_ID),
                        'websites'      => array_keys($this->storeManager->getWebsites(false, true)),
                        'default_store' => $this->storeManager->getDefaultStoreView()->getCode()
                    ]
                ];
            }

            foreach ($mapping as $map) {
                $domainGroup = $this->domainGroupFactory->create();
                $domainGroup
                    ->setDefaultStore($this->storeManager->getStore($map['default_store']))
                    ->setProjectId($map['project_id'])
                    ->setName($map['name']);
                foreach ($map['websites'] as $websiteCode) {
                    $stores = $this->storeManager->getWebsite($websiteCode)->getStores();
                    array_walk($stores, [$domainGroup, 'addStore']);
                }

                $this->mapping[] = $domainGroup;
            }

        }

        return $this->mapping;
    }

    /**
     * @return bool
     */
    public function isMultiDomain(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_FILO_GENERAL_MULTIDOMAIN);
    }

    /**
     * @return array
     * @throws LocalizedException
     */
    public function getProjectIds(): array
    {
        $ids = [];

        foreach ($this->getWebsiteMapping() as $map) {
            $ids[] = $map->getProjectId();
        }

        return array_unique($ids);
    }

    /**
     * @param WebsiteInterface|StoreInterface $scope
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function hasProjectId($scope = null): bool
    {
        foreach ($this->getWebsiteMapping() as $map) {
            if ($map->has($scope)) {
                return true;
            }
        }

        return false;
    }

}
