<?php

declare(strict_types=1);

namespace FiloBlu\Core\Helper;

use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Filesystem;
use Magento\PageBuilder\Controller\Adminhtml\ContentType\Image\Upload;
use Throwable;
use function array_key_exists;
use function in_array;
use Magento\Framework\Exception\FileSystemException;

/**
 *
 */
class WysiwygOptimizer
{

    /**
     * @var Filesystem
     */
    private $filesystem;
    /**
     * @var array
     */
    private $contentProviders;
    /**
     * @var array
     */
    private $contentModifiers;

    /**
     * @param Filesystem $filesystem
     * @param array $contentProviders
     * @param array $contentModifiers
     */
    public function __construct(
        Filesystem $filesystem,
        array      $contentProviders,
        array      $contentModifiers
    )
    {
        $this->filesystem = $filesystem;
        $this->contentProviders = $contentProviders;
        $this->contentModifiers = $contentModifiers;
    }

    /**
     * @return array
     */
    public function getUsedImages()
    {
        $imagesArray = [];

        foreach ($this->contentProviders as $provider) {
            foreach ($provider->getContents() as $content) {
                preg_match_all('/\"(wysiwyg\/.*)\"/U', $content, $output);
                $imagesArray = array_merge($imagesArray, $output[1]);
            }
        }
        return array_unique($imagesArray);
    }

    /**
     * @param array $fileList
     * @return array
     * @throws LocalizedException
     */
    public function checkFiles(array $fileList): array
    {
        $mediaPath = $this->filesystem->getDirectoryRead(DirectoryList::MEDIA);
        $map = [];
        $preservedImage = [];
        $preservedAbsoluteImages = [];

        foreach ($fileList as $file) {
            if (!$mediaPath->isFile($file)) {
                continue;
            }

            if (!$mediaPath->isExist($file)) {
                throw new LocalizedException(
                    __('Could not perform optimization when some file is missing (%1)', $file)
                );
            }

            if (!$mediaPath->isReadable($file)) {
                throw new LocalizedException(
                    __('Could not perform optimization when some file is unreadable (%1)', $file)
                );
            }

            $absoluteFilePath = $mediaPath->getAbsolutePath($file);
            try {
                $md5 = md5_file($absoluteFilePath);
            } catch (Throwable $exception) {
                continue;
            }

            if (!$md5) {
                continue;
            }

            if (!array_key_exists($md5, $map)) {
                $map[$md5] = [];
                $preservedImage[$md5] = $file;
                $preservedAbsoluteImages[$md5] = $absoluteFilePath;
                continue;
            }
            $map[$md5][] = $file;
        }

        return [
            'duplicates' => array_filter($map),
            'preserved' => $preservedImage,
            'preserved_absolute' => $preservedAbsoluteImages
        ];
    }

    /**
     * @param $dataList
     * @throws FileSystemException
     */
    public function moveFiles($dataList)
    {
        $mediaPath = $this->filesystem->getDirectoryWrite(DirectoryList::MEDIA);
        $wysiwygDir = $mediaPath->getAbsolutePath(Upload::UPLOAD_DIR);
        foreach (glob("$wysiwygDir/*") as $file) {
            if (!$mediaPath->isFile($file)) {
                continue;
            }
            if (!$mediaPath->isWritable($file)) {
                echo "Could not perform optimization because this file is not writable $file";
                continue;
            }
            if (in_array($file, $dataList['preserved_absolute'])) {
                continue;
            }
            $mediaPath->renameFile(
                $file,
                str_replace(Upload::UPLOAD_DIR, Upload::UPLOAD_DIR . '/backup', $file)
            );
        }
    }

    /**
     * @param $dataList
     * @return void
     */
    public function fixDuplicatedContentImage($dataList)
    {
        $arrayToReplace = [];

        foreach ($dataList['duplicates'] as $hash => $duplicates) {
            $filename = $dataList['preserved'][$hash];
            foreach ($duplicates as $duplicate) {
                echo "{$duplicate} => {$filename}\n";
                $arrayToReplace[] = ['from' => $duplicate, 'to' => $dataList['preserved'][$hash]];
            }
        }
        foreach ($this->contentModifiers as $modifier) {
            $modifier->updateContent($arrayToReplace);
        }
    }
}
