<?php
declare(strict_types=1);

namespace FiloBlu\Core\Helper\Delivery;

use FiloBlu\Core\Model\Locator\Quote;
use Magento\Eav\Model\Entity\Attribute\AbstractAttribute;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;

class CustomerAddressFieldHelper
{
    const CHECKOUT_FIELDS_ENABLED = 'filoblu_delivery_extension/checkout_fields/enabled';
    const CHECKOUT_FIELDS_DELIVERY_METHODS = 'filoblu_delivery_extension/checkout_fields/delivery_methods';
    const CHECKOUT_FIELDS_CUSTOMER_ADDRESS_REQUIRED_FIELDS = 'filoblu_delivery_extension/checkout_fields/customer_address_required_fields';

    private $scopeConfig;
    private $quoteLocator;

    /**
     * @param ScopeConfigInterface $scopeConfig
     * @param Quote $quoteLocator
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        Quote                $quoteLocator
    )
    {
        $this->scopeConfig = $scopeConfig;
        $this->quoteLocator = $quoteLocator;
    }

    /**
     * @param AbstractAttribute|null $attributeToCheck
     * @return AbstractAttribute|void|null
     */
    public function checkAttribute(?AbstractAttribute $attributeToCheck)
    {
        if (!$this->isCustomFieldEnabled()) {
            return $attributeToCheck;
        }

        if (!$this->quoteLocator->getQuote()) {
            return $attributeToCheck;
        }

        $deliveryMethods = $this->getDeliveryMethods();
        if (empty($deliveryMethods)) {
            return $attributeToCheck;
        }

        $customerAddressRequiredFields = $this->getCustomerAddressRequiredFields();
        if (empty($customerAddressRequiredFields)) {
            return $attributeToCheck;
        }

        $shippingMethod = $this->quoteLocator->getQuote()->getShippingAddress()->getShippingMethod();

        if (!$shippingMethod) {
            return $attributeToCheck;
        }

        $shippingMethodIdentifier = strtok($shippingMethod, '_');

        if (!in_array($shippingMethodIdentifier, $deliveryMethods)) {
            return $attributeToCheck;
        }

        if (!in_array($attributeToCheck->getAttributeCode(), $customerAddressRequiredFields)) {
            return $attributeToCheck;
        }

        return $attributeToCheck->setIsRequired(false);
    }

    /**
     * Is Custom Field Feature enabled
     * @return mixed
     */
    public function isCustomFieldEnabled()
    {
        return $this->scopeConfig->getValue(
            self::CHECKOUT_FIELDS_ENABLED,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * Get Delivery Methods where apply custom logic at checkout
     * @return array|bool|float|int|string|null
     */
    public function getDeliveryMethods()
    {
        $configValue = $this->scopeConfig->getValue(
            self::CHECKOUT_FIELDS_DELIVERY_METHODS,
            ScopeInterface::SCOPE_STORE
        );

        try {
            if ($configValue === null || $configValue === '') {
                return [];
            }
            return explode(',', $configValue);
        } catch (\Exception $e) {
            return [];
        }
    }

    /**
     * Get fields for apply custom logic at checkout
     * @return array|bool|float|int|string|null
     */
    public function getCustomerAddressRequiredFields()
    {
        $configValue = $this->scopeConfig->getValue(
            self::CHECKOUT_FIELDS_CUSTOMER_ADDRESS_REQUIRED_FIELDS,
            ScopeInterface::SCOPE_STORE
        );

        try {
            if ($configValue === null || $configValue === '') {
                return [];
            }
            return explode(',', $configValue);
        } catch (\Exception $e) {
            return [];
        }
    }
}
