<?php

namespace FiloBlu\Core\Framework\Model;

use Magento\Framework\DataObject;

/**
 * Class Mapper
 * @package FiloBlu\Core\Framework\Model
 */
class Mapper
{
    /** @var int  */
    const INT = 1;
    /** @var int  */
    const BOOL = 2;
    /** @var int  */
    const FLOAT = 3;
    /** @var int  */
    const STRING = 4;

    /**
     * @var string null
     */
    protected $currentSourceRoot = null;
    /**
     * @var string null
     */
    protected $currentDestinationRoot = null;
    /**
     * @var array
     */
    protected $map = [];
    /**
     * @var DataSetFactory
     */
    private $dataSetFactory;

    /**
     * Mapper constructor.
     * @param DataSetFactory $dataSetFactory
     */
    public function __construct(DataSetFactory $dataSetFactory)
    {
        $this->dataSetFactory = $dataSetFactory;
    }

    /**
     * @param string $from
     * @param string $to
     * @param null $default
     * @param null $cast
     * @return Mapper
     */
    public function add(string $from, string $to, $default = null, $cast = null)
    {
        if ($this->currentSourceRoot !== null) {
            $from = trim(preg_replace('/\/+/', '/', "{$this->currentSourceRoot}/{$from}"), '/');
        }

        if ($this->currentDestinationRoot !== null) {
            $to = trim(preg_replace('/\/+/', '/', "{$this->currentDestinationRoot}/{$to}"), '/');
        }

        $this->map[$to] = [
            'from'    => $from,
            'default' => $default,
            'cast'    => $cast
        ];
        return $this;
    }

    /**
     * @param DataObject $source
     * @return DataSet
     */
    public function render(DataObject $source)
    {
        /** @var DataSet $data */
        $data = $this->dataSetFactory->create();
        foreach ($this->map as $to => $map) {
            $value = $source->getDataByPath($map['from']);

            if ($value === null) {
                if ($map['default'] !== null) {
                    $value = $map['default'];
                }
            }

            if ($value !== null) {
                switch ($map['cast']) {
                    case self::INT:
                        $value = (int)$value;
                        break;
                    case self::BOOL:
                        $value = (bool)$value;
                        break;
                    case self::FLOAT:
                        $value = (float)$value;
                        break;
                    case self::STRING:
                        $value = (string)$value;
                        break;
                    default:
                }
            }

            $data->setDataByPath($to, $value);
        }

        return $data;
    }

    /**
     * @param string $sourceRoot
     * @return Mapper
     */
    public function withSourceRoot(string $sourceRoot): Mapper
    {
        $this->currentSourceRoot = $sourceRoot;
        return $this;
    }

    /**
     * @param string $destinationRoot
     * @return Mapper
     */
    public function withDestinationRoot(string $destinationRoot): Mapper
    {
        $this->currentDestinationRoot = $destinationRoot;
        return $this;
    }

    /**
     * @return Mapper
     */
    public function resetSourceRoot(): Mapper
    {
        $this->currentSourceRoot = null;
        return $this;
    }

    /**
     * @return Mapper
     */
    public function resetDestinationRoot(): Mapper
    {
        $this->currentDestinationRoot = null;
        return $this;
    }
}
