<?php
declare(strict_types=1);
namespace FiloBlu\Core\Console\Command;

use Exception;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use RuntimeException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Zend_Db_Expr;

/**
 *
 */
class CheckAndFixDuplicateCountryId extends Command
{
    /**
     * @var string
     */
    const NAME = 'name';

    /**
     * @var ResourceConnection
     */
    private $resourceConnection;
    /**
     * @var array
     */
    private $tables;
    /**
     * @var array
     */
    private $defaultNameToExclude;

    /**
     * @param ResourceConnection $resourceConnection
     * @param array $tables
     * @param array $defaultNameToExclude
     * @param string|null $name
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        array              $tables = [],
        array              $defaultNameToExclude = [],
        string             $name = null
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->tables = $tables;
        $this->defaultNameToExclude = $defaultNameToExclude;
        parent::__construct($name);
    }

    /**
     * @return void
     */
    protected function configure()
    {
        $this->setName('filoblu:check:duplicate:region');
        $this->setDescription(
            'Check the duplicates of country_id and remove them updating the correct value in address tables'
        );
        $this->addOption(
            self::NAME,
            null,
            InputOption::VALUE_REQUIRED,
            'Name'
        );
        parent::configure();
    }

    /**
     * Execute the command
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $exitCode = 0;
        $output->writeln('<comment>Checking...</comment>');

        try {
            $this->checkDuplicates($output, $this->getDefaultNamesToExclude());
            $this->updateWrongsRegionId($output);
            $this->checkCountryWithoutMapping($output);
        } catch (Exception $e) {
            $output->writeln(sprintf(
                '<error>%s</error>',
                $e->getMessage()
            ));
            $exitCode = 1;
        }

        $output->writeln([
            '',
            str_repeat('=', 80),
            '<info>Done</info>'
        ]);
        return $exitCode;
    }

    /**
     * @throws Exception
     */
    public function checkDuplicates(OutputInterface $output, array $namesToExclude)
    {
        $connection = $this->resourceConnection->getConnection();
        $table = $connection->getTableName('directory_country_region');

        $select = $connection->select()
            ->from($table, [new Zend_Db_Expr('CONCAT(country_id, \'#\', code)')])
            ->group(['code', 'country_id'])
            ->having(new Zend_Db_Expr('COUNT(*) > 1'))
            ->where('default_name NOT IN(?)', $namesToExclude);

        $duplicates = $connection->fetchCol($select);

        if (!$duplicates) {
            $output->writeln('<info>There are no duplicates</info>');
            return;
        }

        $output->writeln('<comment>Found duplicates</comment>');
        $regionIds = $this->retrieveRegionIds($connection, $duplicates);

        $output->writeln('<comment>Updating addresses tables...</comment>');
        $this->updateAddressesTables($connection, $regionIds);

        return null;
    }

    /**
     * @param \Magento\Framework\DB\Adapter\AdapterInterface $connection
     * @param array $duplicates
     * @return array
     */
    public function retrieveRegionIds(AdapterInterface $connection, array $duplicates): array
    {
        $table = $connection->getTableName('directory_country_region');
        $select = $connection->select()->from(
            $table,
            [
                'region_id' => 'region_id',
                new Zend_Db_Expr('CONCAT(country_id, \'#\', code)')
            ]
        )->where(
            new Zend_Db_Expr('CONCAT(country_id, \'#\', code) IN (?)'),
            $duplicates
        )->order('region_id DESC');
        $regionIds = $connection->fetchPairs($select);

        if (!$regionIds) {
            throw new RuntimeException('Region Ids not found');
        }

        $output = [];
        foreach ($regionIds as $regionId => $key) {
            if (!isset($output[$key])) {
                $output[$key] = ['region_id' => $regionId, 'other_regions' => []];
                continue;
            }
            $output[$key]['other_regions'][] = $regionId;
        }

        return $output;
    }

    /**
     * @param \Magento\Framework\DB\Adapter\AdapterInterface $connection
     * @param array $regionIds
     * @return array
     */
    public function updateAddressesTables(AdapterInterface $connection, array $regionIds): array
    {
        $directoryTable = $connection->getTableName('directory_country_region');

        try {
            $connection->beginTransaction();
            foreach ($regionIds as $regionId) {
                foreach ($this->tables as $table) {
                    $tableToUpdate = $connection->getTableName($table);
                    $dataToUpdate = ['region_id' => $regionId['region_id']];
                    $whereCondition = ['region_id IN(?)' => $regionId['other_regions']];
                    $connection->update($tableToUpdate, $dataToUpdate, $whereCondition);
                }

                $whereCondition = ['region_id IN(?)' => $regionId['other_regions']];
                $connection->delete($directoryTable, $whereCondition);
            }
            $connection->commit();
        } catch (Exception $e) {
            $connection->rollBack();
            throw $e;
        }

        return $regionIds;
    }

    /**
     * @return array
     */
    public function getDefaultNamesToExclude(): array
    {
        return $this->defaultNameToExclude;
    }

    /**
     * @return void
     */
    public function updateWrongsRegionId(OutputInterface $output)
    {
        $connection = $this->resourceConnection->getConnection();
        foreach ($this->tables as $table) {
            $output->writeln([
                '',
                sprintf('<comment>Updating wrong region ids in %s if they exists</comment>', $table)
            ]);
            $select = $connection
                ->select()
                ->join(
                    ['dcr' => 'directory_country_region'],
                    'dcr.default_name = address_table.region ',
                    ['region_id' => 'dcr.region_id']
                )
                ->where('dcr.region_id != address_table.region_id');
            $update = $connection->updateFromSelect($select, ['address_table' => $table]);
            $connection->query($update);
        }
    }

    /**
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     * @return void
     */
    public function checkCountryWithoutMapping(OutputInterface $output)
    {
        $connection = $this->resourceConnection->getConnection();
        $directoryTable = $connection->getTableName('directory_country_region');

        foreach ($this->tables as $table) {
            $output->writeln([
                '',
                '',
                sprintf('<comment>Checking if there are different regions in %s</comment>', $table)
            ]);

            $select = $connection->select();
            $subQuery = $connection->select()
                ->from(['dcr' => $directoryTable], ['default_name']);

            $select->distinct()
                ->from(['address_table' => $connection->getTableName($table)], ['region'])
                ->where('address_table.region NOT IN (?)', $subQuery);

            $distinctRegions = $connection->fetchCol($select);

            if (!$distinctRegions) {
                $output->writeln("<comment>There are no different regions in $table</comment>");
                return;
            }

            $this->printDistinctRegion($distinctRegions, $output);
        }
    }

    /**
     * @param array $distinctRegions
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     * @return void
     */
    public function printDistinctRegion(array $distinctRegions, OutputInterface $output)
    {
        foreach ($distinctRegions as $distinctRegion) {
            $output->writeln("<info>$distinctRegion</info>");
        }
    }
}
