<?php
declare(strict_types=1);

namespace FiloBlu\Core\Block\System\Config\Form\Field;

use Magento\Backend\Block\Template\Context;
use Magento\Config\Block\System\Config\Form\Field\FieldArray\AbstractFieldArray;
use Magento\Framework\Data\Form\Element\AbstractElement;
use Magento\Framework\DataObject;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\View\Element\BlockInterface;
use Magento\Framework\View\Element\Html\Select;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Class Ranges
 */
class WebsiteMapping extends AbstractFieldArray
{
    /**
     * @var BlockInterface
     */
    private $websitesRenderer;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var
     */
    private $storeRenderer;

    /**
     * DynamicSmtp constructor.
     * @param Context $context
     * @param array $data
     */
    public function __construct(Context $context, array $data = [])
    {
        parent::__construct($context, $data);
        $this->storeManager = $context->getStoreManager();
    }

    /**
     * Prepare rendering the new field by adding all the needed columns
     */
    protected function _prepareToRender()
    {
        $this->addColumn('name', ['label' => __('Name'), 'class' => 'required-entry']);

        $this->addColumn('websites', [
            'label'        => __('Websites'),
            'renderer'     => $this->getWebsitesRenderer(),
            'extra_params' => 'multiple="multiple"'
        ]);

        $this->addColumn('default_store', [
            'renderer' => $this->getStoreRenderer(),
            'label'    => __('Default store'),
            'class'    => 'required-entry'
        ]);
        $this->addColumn('project_id', ['label' => __('Project ID'), 'class' => 'required-entry']);
        $this->_addAfter = false;
        $this->_addButtonLabel = __('Add');
    }

    /**
     * @return BlockInterface
     * @throws LocalizedException
     */
    private function getWebsitesRenderer()
    {
        if (!$this->websitesRenderer) {
            $this->websitesRenderer = $this->getLayout()->createBlock(
                WebsiteMultiselect::class,
                '',
                ['data' => ['is_render_to_js_template' => true]]
            )
                ->setId($this->_getCellInputElementId('<%- _id %>', 'websites'))
                ->setName($this->_getCellInputElementName('websites'));
        }
        return $this->websitesRenderer;
    }

    /**
     * @return mixed
     * @throws LocalizedException
     */
    private function getStoreRenderer()
    {
        if (!$this->storeRenderer) {
            $options = [];
            foreach ($this->storeManager->getStores() as $store) {
                $options[] = [
                    'label' => $store->getCode(),
                    'value' => $store->getCode()
                ];
            }
            return $this->getLayout()
                ->createBlock(Select::class)
                ->setId($this->_getCellInputElementId('<%- _id %>', 'default_store'))
                ->setName($this->_getCellInputElementName('default_store'))
                ->setClass('select')
                ->setOptions($options);
        }

        return $this->storeRenderer;
    }

    /**
     * Prepare existing row data object
     *
     * @param DataObject $row
     * @throws LocalizedException
     */
    protected function _prepareArrayRow(DataObject $row): void
    {
        $options = [];

        foreach ($row->getWebsites() ?? [] as $website) {
            $options['option_' . $this->getWebsitesRenderer()->calcOptionHash($website)] = 'selected="selected"';
        }

        $row->setData('option_extra_attrs', $options);
    }

    public function render(AbstractElement $element)
    {
        $isCheckboxRequired = $this->_isInheritCheckboxRequired($element);

        // Disable element if value is inherited from other scope. Flag has to be set before the value is rendered.
        if ($element->getInherit() == 1 && $isCheckboxRequired) {
            $element->setDisabled(true);
        }

        if ($element->getIsDisableInheritance()) {
            $element->setReadonly(true);
        }

        /*   $html = '<td class="label"><label for="' .
               $element->getHtmlId() . '"><span' .
               $this->_renderScopeLabel($element) . '>' .
               $element->getLabel() .
               '</span></label></td>';*/
        $html = $this->_renderValue($element);

        if ($isCheckboxRequired) {
            $html .= $this->_renderInheritCheckbox($element);
        }

        $html .= $this->_renderHint($element);

        return $this->_decorateRowHtml($element, $html);
    }
}
