<?php

namespace FiloBlu\CheckoutExtend\Plugin;

use FiloBlu\CheckoutExtend\Helper\RuleHelper;
use Magento\Sales\Api\OrderManagementInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Exception;
use FiloBlu\CheckoutExtend\Helper\Data;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Customer\Api\Data\CustomerInterfaceFactory;
use Magento\Framework\Event\ManagerInterface;
use Magento\Newsletter\Model\SubscriberFactory;
use Magento\Quote\Model\QuoteFactory;
use Magento\Quote\Model\QuoteRepository;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\SalesRule\Model\Coupon;
use Magento\SalesRule\Model\ResourceModel\Coupon\Usage;
use Magento\SalesRule\Model\Rule\CustomerFactory;
use Psr\Log\LoggerInterface;
use Magento\Customer\Api\AccountManagementInterface;
use Magento\Sales\Model\Order\OrderCustomerExtractor;
use Magento\Customer\Model\Session;
use Magento\Sales\Api\OrderCustomerManagementInterface;
use Magento\Quote\Model\Quote\AddressFactory as QuoteAddressFactory;
use Magento\Framework\App\Http\Context;
use Magento\Customer\Model\CustomerFactory as CustomerModelFactory;
use Magento\Customer\Model\CustomerRegistry;
use Magento\Quote\Api\Data\CartInterfaceFactory;

class OrderPlaceAfterPlugin {

    const DEFAULT_CUSTOMER_GROUP = 'customer/create_account/default_group';

    const CHECKOUT_REGISTRATION_ENABLED = "filoblu_checkoutextend_section/registration/enabled";

    const CHECKOUT_SUBSCRIBER_ENABLED  = "filoblu_checkoutextend_section/newsletter/enabled";

    const CHECKOUT_INVOICE_ENABLED = "filoblu_checkoutextend_section/invoice/enabled";

    const CHECKOUT_DNI_ENABLED = "filoblu_checkoutextend_section/dni/enabled";

    /**
     * @var SubscriberFactory
     */
    protected $_subscriberFactory;
    /**
     * @var Data
     */
    protected $_helper;
    /**
     * @var QuoteFactory
     */
    protected $_quoteFactory;
    /**
     * @var CheckoutSession
     */
    protected $_checkoutSession;
    /**
     * @var CustomerInterfaceFactory
     */
    protected $customerFactory;
    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;
    /**
     * @var ManagerInterface
     */
    protected $_eventManager;
    /**
     * @var Coupon
     */
    protected $coupon;
    /**
     * @var Usage
     */
    protected $couponUsage;
    /**
     * @var LoggerInterface
     */
    protected $logger;
    /**
     * @var CustomerFactory
     */
    protected $ruleCustomerFactory;

    /**
     * @var QuoteRepository
     */
    private $quoteRepository;
    /**
     * @var AccountManagementInterface
     */
    private $accountManagement;
    /**
     * @var OrderCustomerExtractor
     */
    private $customerExtractor;
    /**
     * @var Session
     */
    private $customerSession;
    /**
     * @var OrderCustomerManagementInterface
     */
    private $orderCustomerManagement;
    /**
     * @var QuoteAddressFactory
     */
    private $addressFactory;
    /**
     * @var Context
     */
    private $httpContext;
    /**
     * @var CustomerModelFactory
     */
    private $customerModelFactory;
    /**
     * @var CustomerRegistry
     */
    private $customerRegistry;
    /**
     * @var CartInterfaceFactory
     */
    private $cartInterfaceFactory;
    /**
     * @var RuleHelper
     */
    private $ruleHelper;


    /**
     * OrderPlaceAfterPlugin constructor.
     * @param SubscriberFactory $subscriberFactory
     * @param Data $helper
     * @param QuoteFactory $quoteFactory
     * @param CheckoutSession $checkoutSession
     * @param CustomerInterfaceFactory $customerFactory
     * @param OrderRepositoryInterface $orderRepository
     * @param ManagerInterface $eventManager
     * @param Coupon $coupon
     * @param Usage $couponUsage
     * @param CustomerFactory $ruleCustomerFactory
     * @param LoggerInterface $logger
     * @param AccountManagementInterface $accountManagement
     * @param QuoteRepository $quoteRepository
     * @param OrderCustomerExtractor $customerExtractor
     * @param Session $customerSession
     * @param OrderCustomerManagementInterface $orderCustomerManagement
     * @param QuoteAddressFactory $addressFactory
     * @param Context $httpContext
     * @param CustomerModelFactory $customerModelFactory
     * @param CustomerRegistry $customerRegistry
     * @param CartInterfaceFactory $cartInterfaceFactory
     * @param RuleHelper $ruleHelper
     */
    public function __construct(
        SubscriberFactory $subscriberFactory,
        Data $helper,
        QuoteFactory $quoteFactory,
        CheckoutSession $checkoutSession,
        CustomerInterfaceFactory $customerFactory,
        OrderRepositoryInterface $orderRepository,
        ManagerInterface $eventManager,
        Coupon $coupon,
        Usage $couponUsage,
        CustomerFactory $ruleCustomerFactory,
        LoggerInterface $logger,
        AccountManagementInterface $accountManagement,
        QuoteRepository $quoteRepository,
        OrderCustomerExtractor $customerExtractor,
        Session  $customerSession,
        OrderCustomerManagementInterface $orderCustomerManagement,
        QuoteAddressFactory $addressFactory,
        Context $httpContext,
        CustomerModelFactory $customerModelFactory,
        CustomerRegistry $customerRegistry,
        CartInterfaceFactory $cartInterfaceFactory,
        RuleHelper $ruleHelper
    )
    {
        $this->_subscriberFactory = $subscriberFactory;
        $this->_helper = $helper;
        $this->_quoteFactory = $quoteFactory;
        $this->_checkoutSession = $checkoutSession;
        $this->customerFactory = $customerFactory;
        $this->orderRepository = $orderRepository;
        $this->_eventManager = $eventManager;
        $this->coupon = $coupon;
        $this->couponUsage = $couponUsage;
        $this->ruleCustomerFactory = $ruleCustomerFactory;
        $this->logger = $logger;
        $this->quoteRepository = $quoteRepository;
        $this->accountManagement = $accountManagement;
        $this->customerExtractor = $customerExtractor;
        $this->customerSession = $customerSession;
        $this->orderCustomerManagement = $orderCustomerManagement;
        $this->addressFactory = $addressFactory;
        $this->httpContext = $httpContext;
        $this->customerModelFactory = $customerModelFactory;
        $this->customerRegistry = $customerRegistry;
        $this->cartInterfaceFactory = $cartInterfaceFactory;
        $this->ruleHelper = $ruleHelper;
    }

    /**
     * @param OrderManagementInterface $subject
     * @param OrderInterface $order
     * @return OrderInterface
     */
    public function afterPlace(OrderManagementInterface $subject, OrderInterface $order) {
        try{
            if(!$order->getEntityId()){
                return $order;
            }
            $quote = $this->_quoteFactory->create()->loadByIdWithoutStore($order->getQuoteId());
            $orderIsUpdated= false;
            //perform extra action on order
            //save invoice and DNI
            if ($this->isInvoiceEnabled($order)) {
                if ($quote) {
                    $order->setCheckoutInvoice($quote->getCheckoutInvoice());
                    $order->setCheckoutInvoiceFiscalCode($quote->getCheckoutInvoiceFiscalCode());
                    $order->setCheckoutInvoiceVatNumber($quote->getCheckoutInvoiceVatNumber());
                    $order->setCheckoutInvoiceBusinessName($quote->getCheckoutInvoiceBusinessName());
                    $order->setCheckoutInvoicePec($quote->getCheckoutInvoicePec());
                    $order->setCheckoutInvoiceSdi($quote->getCheckoutInvoiceSdi());
                    $orderIsUpdated =true;
                }

                if ($this->isDniEnabled($order)) {
                    $order->setDni($quote->getDni());
                    $orderIsUpdated =true;
                }
            }

            //convert Guest to Customer
            if(!$order->getCustomerId() && $this->isCheckoutRegistrationEnabled($order)) {
                $customer = $this->customerFactory->create();
                $customer->setStoreId($order->getStoreId());
                if ($order->getEntityId() && !$customer->getId()) {
                    //create Account
                    $customerPwd = $this->_checkoutSession->getRegistrationPwd();
                    $customer = $this->customerExtractor->extract($order->getEntityId());
                    try{
                        $account = $this->accountManagement->createAccount($customer, $customerPwd);
                        $this->customerRegistry->push($this->customerModelFactory->create()->load($account->getId()));
                        $this->_eventManager->dispatch(
                            'customer_register_success_checkout',
                            ['account_controller' => null, 'customer' => $account]
                        );
                        $order->setCustomerId($account->getId());
                        $order->setCustomerGroupId($account->getGroupId());
                        $order->setCustomerIsGuest(0);
                        $orderIsUpdated =true;

                        //Writes Coupon Usages
                        if (!empty($order->getCouponCode())) {
                            $this->updateCouponUsages($order, (int)$account->getId());
                        }

                        //Writes Rule usages
                        if (!empty($order->getAppliedRuleIds())) {
                            $this->ruleHelper->addCustomerRule($order, (int)$account->getId());
                        }
                        //logIn new customer
                        $this->customerSession->regenerateId();
                        $newQuote =$this->cartInterfaceFactory->create()->setCustomer($account)->setIsActive(true)->setStoreId($order->getStoreId())->save();
                        $this->_checkoutSession->setQuoteId($newQuote->getId());
                        $this->customerSession->setCustomerId($account->getId());
                        $this->httpContext->setValue(\Magento\Customer\Model\Context::CONTEXT_AUTH, true, false);
                        $this->customerSession->setCustomerData($account);
                        $customerModel = $this->customerModelFactory->create()->updateData($account);
                        $this->customerSession->setCustomer($customerModel);
                    }catch (\Exception $e){
                        //customer already exists issue . Continue to use guest order
                    }catch (\Throwable $t){
                        //customer already exists issue.  Continue to use guest order
                    }

                }
            }
            //save order if updated
            if($orderIsUpdated){
                $this->orderRepository->save($order);
            }
            $customerId = $this->orderRepository->get($order->getEntityId())->getCustomerId();
            // Subscribe to Newsletter
            if ($this->isCheckoutNewsletterEnabled($order)) {
                if ($customerId) {
                    $this->_subscriberFactory->create()->subscribeCustomerById($customerId);
                } else {
                    $this->_subscriberFactory->create()->subscribe($order->getCustomerEmail());
                }
                $this->_eventManager->dispatch(
                    'filoblu_subscriber_creation_success_checkout', ['order' => $order]
                );
            }
            return $order;
        }catch (\Exception $e){
            $this->logger->critical($e->getMessage());
            return $order;
        }catch (\Throwable $t){
            $this->logger->critical($t->getMessage());
            return $order;
        }
    }

    /**
     * @param OrderInterface $order
     * @return bool
     */
    public function isCheckoutRegistrationEnabled(OrderInterface $order)
    {
        try {
            $quote = $this->quoteRepository->get($order->getQuoteId());
            return ($this->_helper->getConfigValue(self::CHECKOUT_REGISTRATION_ENABLED) &&  $quote->getCheckoutRegistration() && !$this->customerSession->isLoggedIn() && $order->getCustomerIsGuest());
        }catch (\Exception $e){
            $this->logger->critical($e->getMessage());
            return false;
        }catch (\Throwable $t){
            $this->logger->critical($t->getMessage());
            return false;
        }

    }

    /**
     * @param OrderInterface $order
     * @return bool
     */
    public function isCheckoutNewsletterEnabled(OrderInterface $order)
    {
        try{
            $quote = $this->quoteRepository->get($order->getQuoteId());
            return ($this->_helper->getConfigValue(self::CHECKOUT_SUBSCRIBER_ENABLED) && $quote->getIsSubscribeForNewsletter());
        }catch (\Exception $e){
            $this->logger->critical($e->getMessage());
            return false;
        }catch (\Throwable $t){
            $this->logger->critical($t->getMessage());
            return false;
        }
    }

    /**
     * @param OrderInterface $order
     * @return bool
     */
    public function isInvoiceEnabled(OrderInterface $order){
        try{
            $quote = $this->quoteRepository->get($order->getQuoteId());
            return ($this->_helper->getConfigValue(self::CHECKOUT_INVOICE_ENABLED) && $quote->getCheckoutInvoice());
        }catch (\Exception $e){
            $this->logger->critical($e->getMessage());
            return false;
        }catch (\Throwable $t){
            $this->logger->critical($t->getMessage());
            return false;
        }

    }

    /**
     * @param OrderInterface $order
     * @return bool
     */
    public function isDniEnabled(OrderInterface $order){
        try{
            $quote = $this->quoteRepository->get($order->getQuoteId());
            return ($this->_helper->getConfigValue(self::CHECKOUT_DNI_ENABLED) && $quote->getDni());
        }catch (\Exception $e){
            $this->logger->critical($e->getMessage());
            return false;
        }catch (\Throwable $t){
            $this->logger->critical($t->getMessage());
            return false;
        }

    }

    /**
     * @param $subject
     * @param $customerId
     */
    private function updateCouponUsages($subject, $customerId)
    {
        $this->coupon->load($subject->getCouponCode(), 'code');
        if ($this->coupon->getId()) {
            $this->couponUsage->updateCustomerCouponTimesUsed($customerId, $this->coupon->getId());
        }
    }
}

