<?php

namespace FiloBlu\AttributeRelated\Block;

use Magento\Catalog\Block\Product\ImageBuilder;
use Magento\Catalog\Helper\Image;
use Magento\Catalog\Helper\ImageFactory;
use Magento\Catalog\Model\Product\Attribute\Repository;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ResourceModel\Product\Collection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\CatalogInventory\Helper\Stock;
use Magento\CatalogPermissions\App\ConfigInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ProductMetadataInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Magento\Framework\View\Element\Template;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Swatches\Helper\Data;
use Magento\Swatches\Helper\Media;
use Magento\Swatches\Model\ResourceModel\Swatch\CollectionFactory as SwatchCollectionFactory;
use Magento\Swatches\Model\Swatch;

/**
 * Class AttributeRelated
 * @package FiloBlu\AttributeRelated\Block
 */
class AttributeRelated extends Template
{
    /**
     * @var string
     */
    const ALLOWED_PRODUCT_TYPE_PATH = 'filoblu_attributerelated/general/allowed_product_types';
    /**
     * @var string
     */
    const PRODUCT_LIMIT_PATH = 'filoblu_attributerelated/general/product_limit';
    /**
     * @var string
     */
    const ENABLED_ATTRIBUTE_RELATED_PATH = 'filoblu_attributerelated/general/enabled_attribute_related';
    /**
     * @var string
     */
    const ENABLED_ATTRIBUTE_RELATED_LISTING_PATH = 'filoblu_attributerelated/general/enabled_attribute_related_category';
    /**
     * @var string
     */
    const ATTRIBUTE_LABEL_PATH = 'filoblu_attributerelated/general/swatch_attribute_label';
    /**
     * @var string
     */
    const SWATCH_ATTRIBUTE_PATH = 'filoblu_attributerelated/general/swatch_attribute';
    /**
     * @var string
     */
    const ATTRIBUTE_RELATION_PATH = 'filoblu_attributerelated/general/attribute_relation';
    /**
     * @var string
     */
    const HIDE_OUT_OF_STOCK_PATH = 'filoblu_attributerelated/general/hide_out_of_stock';
    /**
     * @var string
     */
    const ENABLED_CURRENT_COLOR_NAME = 'filoblu_attributerelated/general/enabled_current_color_name';
    /**
     * @var string
     */
    const ENABLED_ALL_COLOR_NAME = 'filoblu_attributerelated/general/enabled_all_color_name';
    /**
     * @var string
     */
    const COLOR_NAME_ATTRIBUTE_CODE = 'filoblu_attributerelated/general/color_name_attribute_code';
    /**
     * @var
     */
    protected $currentProduct;

    /**
     * @var StoreManagerInterface
     */
    protected $imageBuilder;
    /**
     * @var Registry
     */
    protected $registry;
    /**
     * @var ProductMetadataInterface
     */
    protected $productMetadata;
    /**
     * @var CollectionFactory
     */
    protected $productCollectionFactory;
    /**
     * @var Stock
     */
    protected $stockFilter;
    /**
     * @var Media
     */
    protected $swatchHelperMedia;
    /**
     * @var array
     */
    protected $lookupWatch = [];
    /**
     * @var Image|mixed
     */
    protected $imageHelper;
    /**
     * @var string
     */
    protected $defaultSwatchThumb;
    /**
     * @var string
     */
    protected $defaultSwatchImage;
    /**
     * @var SwatchCollectionFactory
     */
    protected $swatchCollectionFactory;

    /**
     * @var Repository
     */
    protected $attributeRepository;
    /**
     * @var ImageFactory
     */
    protected $imageHelperFactory;
    /**
     * @var Data
     */
    protected $swatchHelper;
    /**
     * @var Visibility
     */
    protected $productVisibility;
    /**
     * @var Status
     */
    protected $productStatus;
    /**
     * @var ConfigInterface
     */
    private $categoryPermission;

    /**
     * @var array
     */
    protected $relatedProductCache;


    /**
     * AttributeRelated constructor.
     * @param ScopeConfigInterface $scopeConfig
     * @param ImageBuilder $_imageBuilder
     * @param Registry $registry
     * @param ProductMetadataInterface $productMetadata
     * @param CollectionFactory $productCollectionFactory
     * @param Stock $stockFilter
     * @param Media $swatchHelperMedia
     * @param Repository $attributeRepository
     * @param SwatchCollectionFactory $swatchCollectionFactory
     * @param Template\Context $context
     * @param Data $swatchHelper
     * @param Image $imageHelper
     * @param ImageFactory $imageHelperFactory
     * @param Status $productStatus
     * @param Visibility $productVisibility
     * @param ConfigInterface $categoryPermission
     * @param array $data
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        ImageBuilder $_imageBuilder,
        Registry $registry,
        ProductMetadataInterface $productMetadata,
        CollectionFactory $productCollectionFactory,
        Stock $stockFilter,
        Media $swatchHelperMedia,
        Repository $attributeRepository,
        SwatchCollectionFactory $swatchCollectionFactory,
        Template\Context $context,
        Data $swatchHelper,
        Image $imageHelper,
        ImageFactory $imageHelperFactory,
        Status $productStatus,
        Visibility $productVisibility,
        ConfigInterface $categoryPermission,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->_isScopePrivate = true;
        $this->categoryPermission = $categoryPermission;
        $this->_scopeConfig = $scopeConfig;
        $this->imageBuilder = $_imageBuilder;
        $this->registry = $registry;
        $this->productMetadata = $productMetadata;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->stockFilter = $stockFilter;
        $this->swatchHelperMedia = $swatchHelperMedia;
        $this->attributeRepository = $attributeRepository;
        $this->swatchCollectionFactory = $swatchCollectionFactory;
        $this->imageHelper = $imageHelper;
        $this->swatchHelper = $swatchHelper;
        $this->imageHelperFactory = $imageHelperFactory;
        $this->productStatus = $productStatus;
        $this->productVisibility = $productVisibility;
        $this->relatedProductCache = [];
    }

    /**
     * @return mixed
     */
    public function isAttributeRelatedEnabled()
    {
        return $this->_scopeConfig->getValue(
            self::ENABLED_ATTRIBUTE_RELATED_PATH,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function isListingAttributeRelatedEnabled()
    {
        return $this->_scopeConfig->getValue(
            self::ENABLED_ATTRIBUTE_RELATED_LISTING_PATH,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getAttributeLabel()
    {
        return $this->_scopeConfig->getValue(
            self::ATTRIBUTE_LABEL_PATH,
            ScopeInterface::SCOPE_STORE
        );
    }


    /**
     * @param \Magento\Eav\Model\Entity\Collection\AbstractCollection $productCollection
     */
    public function initAttributeRelatedCollection(\Magento\Eav\Model\Entity\Collection\AbstractCollection $productCollection){

        try{

            if(count($this->relatedProductCache) > 0){
                return;
            }

            $hasCategoryPermission = $this->categoryPermission->isEnabled();
            $productRelationAttributeValueArray = [];
            foreach ($productCollection as $prod){
                $productRelationAttributeValueArray [] = $this->getProductRelationAttributeValue($prod);
            }

            $collection = $this->productCollectionFactory->create()
                ->addAttributeToFilter($this->getRelationshipAttribute(), ['in'=> $productRelationAttributeValueArray])
                ->addAttributeToSelect($this->getRelationshipAttribute())
                ->addAttributeToSelect('name')
                ->addAttributeToSelect('thumbnail')
                ->addAttributeToSelect('image')
                ->addAttributeToSelect('small_image')
                ->addAttributeToFilter('type_id', ['in' => $this->getAllowedProductTypes()])
                ->addAttributeToFilter('status', ['in' => $this->productStatus->getVisibleStatusIds()])
                ->addAttributeToFilter('visibility', ['in' => $this->productVisibility->getVisibleInSiteIds()])
                ->setOrder('main_table.sku')
                ->addStoreFilter();
            $swatchAttribute = $this->getSwatchAttribute();
            if($swatchAttribute){
                $collection->addAttributeToSelect($swatchAttribute);
            }
            if ($this->hideOutOfStock()) {
                $this->stockFilter->addInStockFilterToCollection($collection);
            }
            $relatedCollection = [];
            foreach ($collection->getItems() as $key => $item) {
                if ($hasCategoryPermission && $item->getIsHidden()) {
                    $collection->removeItemByKey($key);
                    continue;
                }

                $item->setData('fb_attribute_related_swatch', $this->lookUpAttribute($item, $item->getData($swatchAttribute)));
                $relatedCollection[$this->getProductRelationAttributeValue($item)][] = $item;

            }
            $this->relatedProductCache = $relatedCollection;
        }catch (\Exception $e){
            $this->_logger->critical($e->getMessage());
        }catch (\Throwable $t){
            $this->_logger->critical($t->getMessage());
        }
    }



    /**
     * @param null $product
     * @return mixed
     */
    public function getAttributeRelatedCollection($product = null)
    {
        try{
            if ($product === null) {
                $product = $this->getCurrentProduct();
            }

            if (empty($product)) {
                return [];
            }

            if(isset($this->relatedProductCache[$this->getProductRelationAttributeValue($product)])){
                return $this->relatedProductCache[$this->getProductRelationAttributeValue($product)];
            }
            $hasCategoryPermission = $this->categoryPermission->isEnabled();

            $collection = $this->productCollectionFactory->create()
                ->addAttributeToFilter($this->getRelationshipAttribute(), $this->getProductRelationAttributeValue($product))
                ->setPageSize($this->getProductLimit())
                ->addAttributeToSelect('name')
                ->addAttributeToSelect('thumbnail')
                ->addAttributeToSelect('image')
                ->addAttributeToSelect('small_image')
                ->addAttributeToFilter('type_id', ['in' => $this->getAllowedProductTypes()])
                ->addAttributeToFilter('status', ['in' => $this->productStatus->getVisibleStatusIds()])
                ->addAttributeToFilter('visibility', ['in' => $this->productVisibility->getVisibleInSiteIds()])
                ->setPageSize($this->getProductLimit())
                ->setOrder('main_table.sku')
                ->addStoreFilter();
            $swatchAttribute = $this->getSwatchAttribute();
            if($swatchAttribute){
                $collection->addAttributeToSelect($swatchAttribute);
            }
            if ($this->hideOutOfStock()) {
                $this->stockFilter->addInStockFilterToCollection($collection);
            }


            $relatedCollection = [];
            foreach ($collection->getItems() as $key => $item) {
                if ($hasCategoryPermission && $item->getIsHidden()) {
                    $collection->removeItemByKey($key);
                    continue;
                }

                $item->setData('fb_attribute_related_swatch', $this->lookUpAttribute($item, $item->getData($swatchAttribute)));
            }
            return $collection;
        }catch (\Exception $e){
            $this->_logger->critical($e->getMessage());
        }catch (\Throwable $t){
            $this->_logger->critical($t->getMessage());
        }
    }

    /**
     * @param $product
     * @return mixed
     */
    public function getListingRelatedCollection($product){

        if(isset($this->relatedProductCache[$this->getProductRelationAttributeValue($product)])){
            return $this->relatedProductCache[$this->getProductRelationAttributeValue($product)];
        }
        return $this->getAttributeRelatedCollection($product)->getItems();
    }

    /**
     * Set Current Product From Registry
     */
    public function getCurrentProduct()
    {
        if ($this->currentProduct) {
            return $this->currentProduct;
        }

        return $this->currentProduct = $this->registry->registry('current_product');
    }

    /**
     * @return mixed
     */

    public function getRelationshipAttribute()
    {
        return $this->_scopeConfig->getValue(
            self::ATTRIBUTE_RELATION_PATH,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getProductRelationAttributeValue($product)
    {
        return $product->getData($this->getRelationshipAttribute());
    }

    /**
     * @return mixed
     */
    public function getAllowedProductTypes()
    {
        return $this->_scopeConfig->getValue(
            self::ALLOWED_PRODUCT_TYPE_PATH,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return int
     */
    public function getProductLimit()
    {
        return (int)$this->_scopeConfig->getValue(
            self::PRODUCT_LIMIT_PATH,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return  bool
     */
    public function hideOutOfStock()
    {
        return $this->_scopeConfig->getValue(
            self::HIDE_OUT_OF_STOCK_PATH,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getSwatchAttribute()
    {
        return $this->_scopeConfig->getValue(
            self::SWATCH_ATTRIBUTE_PATH,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @param $product
     * @param $attributeValue
     * @return array|mixed
     * @throws NoSuchEntityException
     */
    public function lookUpAttribute($product, $attributeValue)
    {
        if ($attributeValue === null) {
            return ['type' => 'image', 'value' => '#'];
        }

        if (empty($this->lookupWatch)) {
            if (!$this->defaultSwatchImage) {
                $this->defaultSwatchImage = $this->imageHelper->getDefaultPlaceholderUrl('swatch_image');
            }

            if (!$this->defaultSwatchThumb) {
                $this->defaultSwatchThumb = $this->imageHelper->getDefaultPlaceholderUrl('swatch_thumb');
            }

            $swatchMediaUrl = $this->swatchHelperMedia->getSwatchMediaUrl() . '/';

            foreach ($this->swatchCollectionFactory->create()->getItems() as $swatch) {
                $value = $swatch->getValue();

                switch ($swatch->getType()) {

                    case Swatch::SWATCH_TYPE_TEXTUAL:
                        $value = $value ?: 'Not set';
                        $type = 'text';
                        break;
                    case Swatch::SWATCH_TYPE_VISUAL_IMAGE:
                        $value = $value ? $swatchMediaUrl . $value : null;
                        $type = 'image';
                        break;
                    case Swatch::SWATCH_TYPE_VISUAL_COLOR:
                        $value = $value ?: '#050505';
                        $type = 'color';
                        break;
                    case  Swatch::SWATCH_TYPE_EMPTY:
                    default:
                        $type = 'image';
                }

                $this->lookupWatch[$swatch->getOptionId()] = ['type' => $type, 'value' => $value];
            }
        }

        if (!isset($this->lookupWatch[$attributeValue])) {
            return ['type' => 'image', 'value' => '#'];
        }

        $output = $this->lookupWatch[$attributeValue];
        $attribute = $this->attributeRepository->get($this->getSwatchAttribute());

        /* Workaround to force populateAdditionalDataEavAttribute call */
        if (!$this->swatchHelper->isVisualSwatch($attribute)) {
            return $output;
        }

        if (empty($output['value'])) {
            if ($attribute->getData('use_product_image_for_swatch')) {
                // TODO: on magento 2.2 use product_thumbnail_image
                //       on magento 2.3 product_swatch_image_large
                // $output['value'] = $this->imageHelperFactory->create()->init($product, 'product_swatch_image_large')->getUrl();
                $mageVersion = (float)$this->productMetadata->getVersion();
                if ($mageVersion >= 2.3) {
                    $output['value'] = $this->imageHelperFactory->create()->init($product, 'product_swatch_image_large')->getUrl();
                } else {
                    $output['value'] = $this->imageHelperFactory->create()->init($product, 'product_thumbnail_image')->getUrl();
                }
            } else {
                $output['value'] = $this->defaultSwatchImage;
            }
        }

        return $output;
    }

    /**
     * @param $product
     * @return string
     */
    public function getImageUrl($product){
        $mageVersion = (float)$this->productMetadata->getVersion();
        if ($mageVersion >= 2.3) {
            return $this->imageHelperFactory->create()->init($product, 'product_swatch_image_large')->getUrl();
        } else {
            return $this->imageHelperFactory->create()->init($product, 'product_thumbnail_image')->getUrl();
        }
    }

    /**
     * @return mixed
     */
    public function getColorNameAttributeCode()
    {
        return $this->_scopeConfig->getValue(
            self::COLOR_NAME_ATTRIBUTE_CODE,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function isEnabledDisplayCurrentColorName()
    {
        return $this->_scopeConfig->getValue(
            self::ENABLED_CURRENT_COLOR_NAME,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function isEnabledDisplayAllColorName()
    {
        return $this->_scopeConfig->getValue(
            self::ENABLED_ALL_COLOR_NAME,
            ScopeInterface::SCOPE_STORE
        );
    }
}
