<?php

namespace FiloBlu\AdyenExchangeRate\Model;

use FiloBlu\AdyenExchangeRate\Api\ExchangeRateRepositoryInterface;
use FiloBlu\AdyenExchangeRate\Api\Data\ExchangeRateInterface;
use FiloBlu\AdyenExchangeRate\Model\ExchangeRateFactory;
use FiloBlu\AdyenExchangeRate\Model\ResourceModel\ExchangeRate\CollectionFactory;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\Api\SortOrder;

class ExchangeRateRepository implements \FiloBlu\AdyenExchangeRate\Api\ExchangeRateRepositoryInterface {

    protected $objectFactory;
    protected $collectionFactory;
    protected $searchResultsFactory;

    public function __construct(
            ExchangeRateFactory $objectFactory,
            CollectionFactory $collectionFactory,
            SearchResultsInterfaceFactory $searchResultsFactory
    ) {
        $this->objectFactory = $objectFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
    }

    public function save(ExchangeRateInterface $object) {
        try {
            $object->save();
        } catch (\Exception $e) {
            throw new CouldNotSaveException(__($e->getMessage()));
        }
        return $object;
    }

    public function getById($id) {
        $object = $this->objectFactory->create();
        $object->load($id);
        if (!$object->getId()) {
            throw new NoSuchEntityException(__('Object with id "%1" does not exist.', $id));
        }
        return $object;
    }

    public function delete(ExchangeRateInterface $object) {
        try {
            $object->delete();
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    public function deleteById($id) {
        return $this->delete($this->getById($id));
    }

    public function getList(SearchCriteriaInterface $criteria) {
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $collection = $this->collectionFactory->create();
        foreach ($criteria->getFilterGroups() as $filterGroup) {
            $fields = [];
            $conditions = [];
            foreach ($filterGroup->getFilters() as $filter) {
                $condition = $filter->getConditionType() ? $filter->getConditionType() : 'eq';
                $fields[] = $filter->getField();
                $conditions[] = [$condition => $filter->getValue()];
            }
            if ($fields) {
                $collection->addFieldToFilter($fields, $conditions);
            }
        }
        $searchResults->setTotalCount($collection->getSize());
        $sortOrders = $criteria->getSortOrders();
        if ($sortOrders) {
            /** @var SortOrder $sortOrder */
            foreach ($sortOrders as $sortOrder) {
                $collection->addOrder(
                        $sortOrder->getField(), ($sortOrder->getDirection() == SortOrder::SORT_ASC) ? 'ASC' : 'DESC'
                );
            }
        }
        $collection->setCurPage($criteria->getCurrentPage());
        $collection->setPageSize($criteria->getPageSize());
        $objects = [];
        foreach ($collection as $objectModel) {
            $objects[] = $objectModel;
        }
        $searchResults->setItems($objects);
        return $searchResults;
    }

    public function getByDate($date = '1980-01-31') {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $searchCriteriaBuilder = $objectManager->create('Magento\Framework\Api\Search\SearchCriteriaBuilder');
        $filter = $objectManager->create('Magento\Framework\Api\Filter');
        $filter->setField('day_date')->setValue($date);
        $searchCriteria = $searchCriteriaBuilder->addFilter($filter)->addSortOrder('currency', 'ASC')->create();
        $results = $this->getList($searchCriteria);
        return $results;
    }

}
