<?php

namespace FiloBlu\AdyenExchangeRate\Model\Currency\Import;

use FiloBlu\AdyenExchangeRate\Model\ExchangeRateRepository;
use FiloBlu\AdyenExchangeRate\Model\Retriever;

/**
 * Currency rate import model (From Adyen currency exchange rate report)
 */
class Adyen extends \Magento\Directory\Model\Currency\Import\AbstractImport
{

    /**
     * Core scope config
     *
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var Repository
     */
    private $repository;

    /**
     * @var Retriever
     */
    private $retriever;

    /**
     * Initialize dependencies
     *
     * @param \Magento\Directory\Model\CurrencyFactory $currencyFactory
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     */
    public function __construct(
        \Magento\Directory\Model\CurrencyFactory $currencyFactory,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        ExchangeRateRepository $repository,
        Retriever $retriever
    ) {
        parent::__construct($currencyFactory);
        $this->scopeConfig = $scopeConfig;
        $this->repository = $repository;
        $this->retriever = $retriever;
    }

    /**
     * {@inheritdoc}
     */
    public function fetchRates()
    {
        $data = [];

        try {
            $currencies = $this->_getCurrencyCodes();
            $defaultCurrencies = $this->_getDefaultCurrencyCodes();

            /*
             * Get yesterday currency rate, due to adyen modifications.
             * Today rate will be available tomorrow.
             */
            $now = new \DateTime();
            $requestedDate = $now->sub(new \DateInterval('P1D'));
            $requestedDate = $requestedDate->format('Y-m-d');
            $rates = $this->repository->getByDate($requestedDate)->getItems();
            if (empty($rates)) {
                $this->retriever->retrieveDayRates($requestedDate);
                $rates = $this->repository->getByDate($requestedDate)->getItems();
            }

            //first get rates with EUR as base
            $eur_index = array_search('EUR', $defaultCurrencies);
            if ($eur_index !== FALSE) {
                $data = $this->convertBatch($data, 'EUR', $currencies, $rates);
                unset($defaultCurrencies[$eur_index]);
            }

            //then use the EUR-based rates to calculate rates with other bases
            //typically we don't need them, but Magento needs them
            foreach ($defaultCurrencies as $currencyFrom) {
                $data = $this->convertBatch($data, $currencyFrom, $currencies, $rates);
            }
        } catch (\Exception $e) {
            $this->_messages[] = $e->getMessage();
        }

        return $data;
    }


    /**
     * Return currencies convert rates in batch mode
     *
     * @param array $data
     * @param string $currencyFrom
     * @param array $currenciesTo
     * @param array $rates
     * @return array
     */
    private function convertBatch($data, $currencyFrom, $currenciesTo, $rates)
    {

        if ($currencyFrom == 'EUR') {
            foreach ($currenciesTo as $currencyTo) {
                if ($currencyFrom == $currencyTo) {
                    $data[$currencyFrom][$currencyTo] = $this->_numberFormat(1);
                } else {
                    $rate = $this->rateLookup($rates, $currencyTo);
                    $data[$currencyFrom][$currencyTo] = $this->_numberFormat($rate['rate']);
                }
            }
        } else {
            foreach ($currenciesTo as $currencyTo) {
                if ($currencyFrom == $currencyTo) {
                    $data[$currencyFrom][$currencyTo] = $this->_numberFormat(1);
                } elseif ($currencyTo == 'EUR' && isset($data['EUR'][$currencyFrom])) {
                    $data[$currencyFrom][$currencyTo] = round(1 / $data['EUR'][$currencyFrom], 4);
                } else {
                    //$this->_messages[] = __('We can\'t retrieve a rate from %1 for %2.', $currencyFrom, $currencyTo);
                    $data[$currencyFrom][$currencyTo] = 1;
                }
            }
        }
        return $data;
    }


    private function rateLookup($rates, $currency) {
        foreach ($rates as $rate) {
            if ($rate['currency'] == $currency) {
                return $rate;
            }
        }
        return NULL;
    }


    /**
     * {@inheritdoc}
     */
    protected function _convert($currencyFrom, $currencyTo)
    {
    }
}
