<?php

namespace FiloBlu\AdyenExchangeRate\Console;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Output\OutputInterface;
use FiloBlu\AdyenExchangeRate\Model\Retriever;
use Magento\Framework\App\State;

class RetrieveYearCommand extends Command
{
    /**
     * @var Retriever
     */
    private $retriever;

    /**
     * @var State
     */
    private $state;

    const INPUT_KEY_YEAR = 'year';
    const INPUT_KEY_CURRENCY = 'currency';

    /**
     * Constructor
     * @param Retriever $retriever
     * @param State $state
     */
    public function __construct(
        Retriever $retriever,
        State $state
    ) {
        $this->retriever = $retriever;
        $this->state = $state;
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setName('filoblu:adyenexchangerate:retrieve-year')
            ->setDescription('Retrieve currency exchange rates for a given year and currency')
            ->setDefinition($this->getInputList());
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $year = $input->getArgument(self::INPUT_KEY_YEAR);
        if (!$year || !is_numeric($year)) {
            $output->writeln('<error>Please provide a valid year.</error>');
            return 1;
        }

        $currency = $input->getArgument(self::INPUT_KEY_CURRENCY);
        if (!in_array($currency, $this->retriever->getCurrencies())) {
            $output->writeln('<error>Invalid currency specified. Please use one of: ' . implode(', ', $this->retriever->getCurrencies()) . '</error>');
            return 1;
        }

        $this->state->setAreaCode('adminhtml');

        $startDate = new \DateTime("{$year}-01-01");

        $today = new \DateTime();
        $currentYear = $today->format('Y');

        if ($year == $currentYear) {
            $endDate = $today;
        } elseif ($year > $currentYear) {
            $output->writeln('<error>Cannot retrieve rates for a future year.</error>');
            return 1;
        } else {
            $endDate = new \DateTime("{$year}-12-31");
        }

        $currentDate = clone $startDate;

        while ($currentDate <= $endDate) {
            $dateString = $currentDate->format('Y-m-d');
            $output->writeln("Retrieving rates for {$dateString} and currency {$currency}...");
            try {
                $this->retriever->retrieveDayRates($dateString, $currency);
                $output->writeln("<info>Successfully retrieved rates for {$dateString} and currency {$currency}.</info>");
            } catch (\Exception $e) {
                $output->writeln("<error>Error retrieving rates for {$dateString}: {$e->getMessage()}</error>");
            }
            $currentDate->modify('+1 day');
        }

        $output->writeln('<info>Finished retrieving rates for the specified period.</info>');
        return 0;
    }

    /**
     * Get list of options and arguments for the command
     *
     * @return mixed
     */
    public function getInputList()
    {
        return [
            new InputArgument(
                self::INPUT_KEY_YEAR,
                InputArgument::REQUIRED,
                'Year to retrieve the currency exchange rates for.'
            ),
            new InputArgument(
                self::INPUT_KEY_CURRENCY,
                InputArgument::REQUIRED,
                'Currency to retrieve rates for. Available currencies: ' . implode(', ', $this->retriever->getCurrencies())
            ),
        ];
    }
}
