<?php

namespace FiloBlu\AdyenExchangeRate\Console;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Helper\ProgressBar;
use FiloBlu\AdyenExchangeRate\Model\ExchangeRateRepository;
use FiloBlu\AdyenExchangeRate\Model\Retriever;
use FiloBlu\AdyenExchangeRate\Model\OrderProcessor;

class ProcessCommand extends Command
{
    /**
     * @var Repository
     */
    private $repository;

    /**
     * @var Retriever
     */
    private $retriever;

    /**
     * @var OrderProcessor
     */
    private $orderProcessor;

    /**
     *
     * @var \Magento\Framework\App\State
     */
    private $state;

    const INPUT_KEY_DATE = 'date';

    const INPUT_TYPE_TO_UPDATE = 'update_type';

    /**
     * Constructor
     * @param ExchangeRateRepository $repository
     */
    public function __construct(
            ExchangeRateRepository $repository,
            Retriever $retriever,
            \Magento\Framework\App\State $state,
            OrderProcessor $processor
            )
    {
        $this->repository = $repository;
        $this->retriever = $retriever;
        $this->state = $state;
        $this->orderProcessor = $processor;
        parent::__construct();
    }

    protected function configure()
    {
        $this
                ->setName('filoblu:adyenexchangerate:process')
                ->setDescription('Process orders for a given date. Retrieve rates if not present.')
                ->setDefinition($this->getInputList());
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $updateType = 'order';

        if ($input->getArgument(self::INPUT_KEY_DATE)) {
            $requestedDate = trim($input->getArgument(self::INPUT_KEY_DATE));
        }

        if($input->getArgument(self::INPUT_TYPE_TO_UPDATE) === 'invoice') {
            $updateType = 'invoice';
        }

        $rates = $this->repository->getByDate($requestedDate)->getItems();
        if (empty($rates)) {
            $output->writeln('Retrieving rates, because they were missing.');
            $this->retriever->retrieveDayRates($requestedDate);
            $rates = $this->repository->getByDate($requestedDate)->getItems();
        }

        try {
            $this->state->getAreaCode();
        } catch (\Exception $ex) {
            $this->state->setAreaCode('crontab');
        }

        if ($updateType == 'invoice') {
            $invoices = $this->orderProcessor->getInvoices($requestedDate);
            $output->writeln('Found ' . count($invoices) . ' to be processed.');
            if (!empty($invoices)) {
                $processed = array();
                $progress = new ProgressBar($output, count($invoices));
                foreach ($invoices as $invoice) {
                    $order = $invoice->getOrder();
                    $this->orderProcessor->addCurrencyRate($order, $rates, $invoice);
                    $progress->advance(1);
                    $processed[] = $invoice->getOrder()->getIncrementId();
                }
                $progress->finish();
                $output->writeln('');
                $output->writeln('The following orders were processed: ' . implode(', ', $processed));
            }
        } else {
            $orders = $this->orderProcessor->getOrders($requestedDate);
            $output->writeln('Found ' . $orders->getTotalCount() . ' to be processed.');
            if(!empty($orders)) {
                $processed = array();
                $progress = new ProgressBar($output, $orders->getTotalCount());
                foreach ($orders as $order) {
                    $invoice = null;
                    foreach ($order->getInvoiceCollection() as $val)
                    {
                        if($val->getId())
                        {
                            $invoice = $this->orderProcessor->getInvoiceById($val->getId());
                        }
                    }

                    $this->orderProcessor->addCurrencyRate($order, $rates, $invoice);
                    $progress->advance(1);
                    $processed[] = $order->getIncrementId();
                }
                $progress->finish();
                $output->writeln('');
                $output->writeln('The following orders were processed: ' . implode(', ', $processed));
            }
        }
    }


    /**
     * Get list of options and arguments for the command
     *
     * @return mixed
     */
    public function getInputList()
    {
        return [
            new InputArgument(
                self::INPUT_KEY_DATE,
                InputArgument::REQUIRED,
                'Date to process orders.'
            ),
            new InputArgument(
                self::INPUT_TYPE_TO_UPDATE,
                InputArgument::OPTIONAL,
                'Process by order or invoice.'
            )
        ];
    }
}
