<?php

namespace FiloBlu\AdyenExchangeRate\Model;

use FiloBlu\AdyenExchangeRate\Helper\CustomLogger;
use FiloBlu\AdyenExchangeRate\Model\ExchangeRateRepository;
use FiloBlu\AdyenExchangeRate\Model\Retriever;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Api\InvoiceRepositoryInterface;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\Api\Search\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrder;


/**
 * Assign currency exchange rate to orders
 *
 * @author ermanno
 */
class OrderProcessor extends \Magento\Framework\Model\AbstractModel {

    const HOME_CURRENCY = 'EUR';

    const INPUT_FIX_MODE = 'mode';

    const INPUT_FIX_MODE_DEFAULT = 'collection';

    const PERIOD_FIX_WEEKLY = 'weekly';

    const PERIOD_FIX_ANNUAL = 'annual';

    /**
     * @var Repository
     */
    private $repository;

    /**
     * @var Retriever
     */
    private $retriever;

    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;

    /**
     * @var InvoiceRepositoryInterface
     */
    private $invoiceRepository;

    protected $searchCriteriaBuilder;

    protected $connection;

    /**
     * @var CustomLogger
     */
    protected $customLogger;

    public function __construct(
            \Magento\Framework\Model\Context $context,
            \Magento\Framework\Registry $registry,
            ExchangeRateRepository $repository,
            Retriever $retriever,
            OrderRepositoryInterface $orderRepository,
            InvoiceRepositoryInterface $invoiceRepository,
            SearchResultsInterfaceFactory $searchResultsFactory,
            SearchCriteriaBuilder $searchCriteriaBuilder,
            ScopeConfigInterface $config,
            CustomLogger $customLogger,
            \Magento\Framework\App\ResourceConnection $resourceConnection,
            \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
            \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
            array $data = array()) {
        parent::__construct(
                $context,
                $registry,
                $resource,
                $resourceCollection,
                $data);
        $this->repository = $repository;
        $this->retriever = $retriever;
        $this->orderRepository = $orderRepository;
        $this->invoiceRepository = $invoiceRepository;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->config = $config;
        $this->customLogger = $customLogger;
        $this->connection = $resourceConnection->getConnection();
    }

    public function process($requestedDate)
    {
        $requestedDate = trim($requestedDate);
        $rates = $this->repository->getByDate($requestedDate)->getItems();
        if (empty($rates)) {
            $this->retriever->retrieveDayRates($requestedDate);
            $rates = $this->repository->getByDate($requestedDate)->getItems();
        }
        $orders = $this->getOrders($requestedDate);
        if (!empty($orders)) {
            foreach ($orders as $order) {
                $this->addCurrencyRate($order, $rates);
            }
        }
    }

    public function getInvoices($date = null) {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $searchCriteriaBuilder = $objectManager->create('Magento\Framework\Api\Search\SearchCriteriaBuilder');
        $filterCurrency = $objectManager->create('Magento\Framework\Api\Filter');
        $filterCurrency->setField('order_currency_code')->setValue(self::HOME_CURRENCY)->setConditionType('neq');
        $searchCriteriaBuilder->addFilter($filterCurrency);
        $filterExchangeRate = $objectManager->create('Magento\Framework\Api\Filter');
        $filterExchangeRate->setField('pay_exchangerate')->setValue(0);
        $searchCriteriaBuilder->addFilter($filterExchangeRate);
        if ($date != null) {
            $dateFrom = \DateTime::createFromFormat('Y-m-d', $date);
            $dateTo = clone $dateFrom->setTime(0,0,0);
            $dateTo->add(new \DateInterval('P1D'));
            $filterFrom = $objectManager->create('Magento\Framework\Api\Filter');
            $filterFrom->setField('created_at')->setValue($dateFrom->format('Y-m-d H:i:s'))->setConditionType('gteq');
            $searchCriteriaBuilder->addFilter($filterFrom);
            $filterTo = $objectManager->create('Magento\Framework\Api\Filter');
            $filterTo->setField('created_at')->setValue($dateTo->format('Y-m-d H:i:s'))->setConditionType('lt');
            $searchCriteriaBuilder->addFilter($filterTo);
        }
        $searchCriteria = $searchCriteriaBuilder->addSortOrder('created_at', 'ASC')->create();
        return $this->invoiceRepository->getList($searchCriteria);
    }

    public function getOrders($date = null) {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $searchCriteriaBuilder = $objectManager->create('Magento\Framework\Api\Search\SearchCriteriaBuilder');
        $filterCurrency = $objectManager->create('Magento\Framework\Api\Filter');
        $filterCurrency->setField('order_currency_code')->setValue(self::HOME_CURRENCY)->setConditionType('neq');
        $searchCriteriaBuilder->addFilter($filterCurrency);
        $filterExchangeRate = $objectManager->create('Magento\Framework\Api\Filter');
        $filterExchangeRate->setField('base_to_global_rate')->setValue(0);
        $searchCriteriaBuilder->addFilter($filterExchangeRate);
        if ($date != null) {
            $dateFrom = \DateTime::createFromFormat('Y-m-d', $date);
            $dateTo = clone $dateFrom->setTime(0,0,0);
            $dateTo->add(new \DateInterval('P1D'));
            $filterFrom = $objectManager->create('Magento\Framework\Api\Filter');
            $filterFrom->setField('created_at')->setValue($dateFrom->format('Y-m-d H:i:s'))->setConditionType('gteq');
            $searchCriteriaBuilder->addFilter($filterFrom);
            $filterTo = $objectManager->create('Magento\Framework\Api\Filter');
            $filterTo->setField('created_at')->setValue($dateTo->format('Y-m-d H:i:s'))->setConditionType('lt');
            $searchCriteriaBuilder->addFilter($filterTo);
        }
        $searchCriteria = $searchCriteriaBuilder->addSortOrder('created_at', 'ASC')->create();
        return $this->orderRepository->getList($searchCriteria);
    }

    public function getInvoiceById($id)
    {
        return $this->invoiceRepository->get($id);
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @param type $rates
     * @param \Magento\Sales\Model\Order\Invoice|null $invoice
     * @return bool
     */
    public function addCurrencyRate($order, $rates, $invoice = null) {
        $order_currency = $order->getBaseCurrencyCode();
        $found_rate = FALSE;
        foreach ($rates as $rate_data) {
            if ($rate_data->getCurrency() == $order_currency) {
                $found_rate = $rate_data;
                break;
            }
        }
        if ($found_rate) {
            $payExchangerate = $found_rate->getRate();
            $baseToGlobalRate = 1/$found_rate->getRate();

            $order->setPayExchangerate($payExchangerate);
            $order->setBaseToGlobalRate($baseToGlobalRate);
            if(!is_null($invoice))
            {
                $invoice->setPayExchangerate($payExchangerate);
                $invoice->setBaseToGlobalRate($baseToGlobalRate);
                $this->invoiceRepository->save($invoice);
            }
            $this->orderRepository->save($order);
            return TRUE;
        } else {
            return FALSE;
        }
    }

    /**
     * @param int $orderId
     * @param $rates
     * @param string $orderCurrency
     * @return int
     */
    public function updateCurrencryRate($orderId, $rates, $orderCurrency) {
        $found_rate = FALSE;
        foreach ($rates as $rate_data) {
            if ($rate_data->getCurrency() == $orderCurrency) {
                $found_rate = $rate_data;
                break;
            }
        }
        $affectedRows = 0;
        if ($found_rate) {
            $orderTable = $this->connection->getTableName('sales_order');
            $affectedRows = $this->connection->update(
                $orderTable,
                [
                    'pay_exchangerate' => $found_rate->getRate(),
                    'base_to_global_rate' => 1/$found_rate->getRate()
                ],
                ['entity_id = ?' => $orderId]
            );
        }
        return $affectedRows;
    }

    /**
     * @param $year
     * @param $currency
     * @return \Magento\Sales\Api\Data\OrderSearchResultInterface
     * @throws \Exception
     */
    public function getOrderByYearAndCurrency($year, $currency) {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $searchCriteriaBuilder = $objectManager->create('Magento\Framework\Api\Search\SearchCriteriaBuilder');
        $filterCurrency = $objectManager->create('Magento\Framework\Api\Filter');
        $filterCurrency->setField('order_currency_code')->setValue($currency)->setConditionType('eq');
        $searchCriteriaBuilder->addFilter($filterCurrency);
        $dateFrom = new \DateTime($year."-01-01");
        $dateFrom->setTime(0,0,0);
        $dateTo = new \DateTime(($year+1)."-01-01");
        $dateTo->setTime(0,0,0);
        $filterFrom = $objectManager->create('Magento\Framework\Api\Filter');
        $filterFrom->setField('created_at')->setValue($dateFrom->format('Y-m-d H:i:s'))->setConditionType('gteq');
        $searchCriteriaBuilder->addFilter($filterFrom);
        $filterTo = $objectManager->create('Magento\Framework\Api\Filter');
        $filterTo->setField('created_at')->setValue($dateTo->format('Y-m-d H:i:s'))->setConditionType('lt');
        $searchCriteriaBuilder->addFilter($filterTo);

        $searchCriteria = $searchCriteriaBuilder->addSortOrder('created_at', 'ASC')->create();
        return $this->orderRepository->getList($searchCriteria);
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @return bool
     * @throws \Exception
     */
    public function restoreCurrencyRate($order) {
        $requestedDate = new \DateTime($order->getCreatedAt());
        $requestedDate->setTime(0,0,0);
        $rates = $this->repository->getByDate($requestedDate->format('Y-m-d'))->getItems();
        if(empty($rates)) {
            $this->retriever->retrieveDayRates($requestedDate->format('Y-m-d'));
            $rates = $this->repository->getByDate($requestedDate->format('Y-m-d'))->getItems();
        }
        return $this->addCurrencyRate($order, $rates);
    }

    public function restoreCurrencyRateDirect($orderId, $orderDate, $orderCurrency) {
        $requestedDate = new \DateTime($orderDate);
        $requestedDate->setTime(0,0,0);
        $rates = $this->repository->getByDate($requestedDate->format('Y-m-d'))->getItems();
        if(empty($rates)) {
            $this->retriever->retrieveDayRates($requestedDate->format('Y-m-d'));
            $rates = $this->repository->getByDate($requestedDate->format('Y-m-d'))->getItems();
        }
        return $this->updateCurrencryRate($orderId, $rates, $orderCurrency);
    }

    public function loadDirectOrderByYearAndCurrency($year, $currency) {
        $orderTable = $this->connection->getTableName('sales_order');
        $dateFrom = new \DateTime($year."-01-01");
        $dateFrom->setTime(0,0,0);
        $dateTo = new \DateTime(($year+1)."-01-01");
        $dateTo->setTime(0,0,0);

        $query = "SELECT entity_id, increment_id, created_at, order_currency_code
        FROM {$orderTable}
        WHERE created_at >= '".$dateFrom->format('Y-m-d H:i:s')."'
        AND created_at < '".$dateTo->format('Y-m-d H:i:s')."'
        AND order_currency_code = '".$currency."'
        ORDER BY created_at ASC";

        $orders = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);
        return $orders;
    }

    public function loadDirectLastWeekOrder() {
        $orderTable = $this->connection->getTableName('sales_order');
        $dateTo = new \DateTime();
        $dateTo->setTime(0,0,0);
        $now = new \DateTime();
        $dateFrom = $now->sub(new \DateInterval('P7D'));
        $dateFrom->setTime(0,0,0);

        $query = "SELECT entity_id, increment_id, created_at, order_currency_code
        FROM {$orderTable}
        WHERE created_at >= '".$dateFrom->format('Y-m-d H:i:s')."'
        AND created_at < '".$dateTo->format('Y-m-d H:i:s')."'
        AND order_currency_code != 'EUR'
        ORDER BY created_at ASC";

        $orders = $this->connection->fetchAll($query, [], \Zend_Db::FETCH_OBJ);
        return $orders;
    }

    public function fixOrder($fixType = 'query', $periodToFix = self::PERIOD_FIX_WEEKLY, $requestedYear = null, $currency = null) {

        if($fixType == self::INPUT_FIX_MODE_DEFAULT) {
            $orders = $this->getOrderByYearAndCurrency($requestedYear, $currency);
            if(!empty($orders)){
                $this->customLogger->log('Found ' . $orders->getTotalCount() . ' to be processed.');
                $processed = array();
                foreach ($orders as $order) {
                    $this->restoreCurrencyRate($order);
                    $processed[] = $order->getIncrementId();
                }
                $this->customLogger->log('The following orders were processed: ' . implode(', ', $processed));
            }
        } elseif($fixType === 'query') {
            if($periodToFix == self::PERIOD_FIX_ANNUAL) {
                $orders = $this->loadDirectOrderByYearAndCurrency($requestedYear, $currency);
            } else {
                $orders = $this->loadDirectLastWeekOrder();
            }
            if(!empty($orders)) {
                $this->customLogger->log('Found ' . count($orders) . ' orders to be processed.');
                $processed = array();
                foreach ($orders as $order) {
                    $this->restoreCurrencyRateDirect($order->entity_id, $order->created_at, $order->order_currency_code);
                    $processed[] = $order->increment_id;
                }
                $this->customLogger->log('The following orders were processed: ' . implode(', ', $processed));
            } else {
                $this->customLogger->log('There are no orders that need to be fixed.');
            }
        } else {
            throw new \Exception('Wrong fix type!');
        }
    }
}
