<?php

namespace FiloBlu\AdyenExchangeRate\Model;

use Exception;
use FiloBlu\AdyenExchangeRate\Api\RetrieverInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;

/**
 * Download currency exchange rates for a given day and save them in its db table
 *
 * @author Ermanno
 */
class Retriever extends AbstractModel implements RetrieverInterface
{

    const REPORT_URL = "https://ca-live.adyen.com/reports/download/MerchantAccount/{merchant_account}/exchange_rate_report_";
    const ENABLE_CONFIG_PATH = 'filoblu_adyenexchangerate/general/active';
    const MERCHANT_ACCOUNT_CONFIG_PATH = 'filoblu_adyenexchangerate/general/merchant_account';
    const LOGIN_CONFIG_PATH = 'filoblu_adyenexchangerate/general/ws_login';
    const PASSWORD_CONFIG_PATH = 'filoblu_adyenexchangerate/general/ws_password';
    const PRIMARY_CURRENCY_MARKUP = 0.012; //1.2%
    const EXOTIC_CURRENCY_MARKUP = 0.03; //3%
    protected $ws_user;

    //Adyen markups on currency conversion
    protected $ws_pwd;
    protected $ws_host;
    protected $currencies = array(
        'USD' => self::PRIMARY_CURRENCY_MARKUP,
        'GBP' => self::PRIMARY_CURRENCY_MARKUP,
        'CHF' => self::PRIMARY_CURRENCY_MARKUP,
        'JPY' => self::PRIMARY_CURRENCY_MARKUP,
        'RUB' => self::EXOTIC_CURRENCY_MARKUP,
        'CNY' => self::EXOTIC_CURRENCY_MARKUP,
        'HKD' => self::PRIMARY_CURRENCY_MARKUP,
        'AED' => self::EXOTIC_CURRENCY_MARKUP,
        'DKK' => self::PRIMARY_CURRENCY_MARKUP,
        'NOK' => self::PRIMARY_CURRENCY_MARKUP,
        'SEK' => self::PRIMARY_CURRENCY_MARKUP,
        'CAD' => self::PRIMARY_CURRENCY_MARKUP,
        'AUD' => self::PRIMARY_CURRENCY_MARKUP,
        'MOP' => self::EXOTIC_CURRENCY_MARKUP,
        'PLN' => self::PRIMARY_CURRENCY_MARKUP,
        'TWD' => self::EXOTIC_CURRENCY_MARKUP,
        'KRW' => self::EXOTIC_CURRENCY_MARKUP,
        'BGN' => self::PRIMARY_CURRENCY_MARKUP,
        'CZK' => self::PRIMARY_CURRENCY_MARKUP,
        'RON' => self::PRIMARY_CURRENCY_MARKUP,
        'HUF' => self::PRIMARY_CURRENCY_MARKUP
    );

    /**
     * @var Curl
     */
    protected $_curl;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var ExchangeRateFactory
     */
    protected $exchangeRateFactory;

    /**
     * @var ExchangeRateRepository
     */
    protected $exchangeRateRepository;

    public function __construct(
        Context                   $context,
        ScopeConfigInterface $scopeConfig,
        Registry                        $registry,
        Curl                $curl,
        ExchangeRateFactory                                $exchangeRateFactory,
        ExchangeRateRepository                             $exchangeRateRepository,
        AbstractResource                                   $resource = null,
        AbstractDb                                         $resourceCollection = null,
        array                                              $data = array())
    {
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
        $this->scopeConfig = $scopeConfig;
        $this->_curl = $curl;
        $this->exchangeRateFactory = $exchangeRateFactory;
        $this->exchangeRateRepository = $exchangeRateRepository;
    }

    public function retrieveDayRates($date = '1980-01-31')
    {
        if ($date == '1980-01-31') {
            throw new Exception('Missing input date to retrieve exchange rates');
        }
        if (!$this->scopeConfig->isSetFlag(self::ENABLE_CONFIG_PATH)) {
            return TRUE;
        }

        $rates_raw_data = $this->getDataFromWS($date);

        $parsed_rates = $this->parseRateData($rates_raw_data, $date);

        //save into db
        foreach ($parsed_rates as $symbol => $rate) {
            $rate_model = $this->exchangeRateFactory->create();
            $rate_model->setData($rate);
            $this->exchangeRateRepository->save($rate_model);
        }

        return TRUE;
    }


    protected function getDataFromWS($date)
    {
        $url = str_replace(
            "{merchant_account}",
            $this->scopeConfig->getValue(self::MERCHANT_ACCOUNT_CONFIG_PATH),
            self::REPORT_URL);

        $date_fragments = explode("-", $date);
        $filename = $date_fragments[0] . '_' . $date_fragments[1] . '_' . $date_fragments[2] . '.csv';
        $url .= $filename;

        $this->_curl->setCredentials(
            $this->scopeConfig->getValue(self::LOGIN_CONFIG_PATH),
            $this->scopeConfig->getValue(self::PASSWORD_CONFIG_PATH));
        $this->_curl->get($url);
        if ($this->_curl->getStatus() != 200) {
            throw new Exception('Cannot access to Adyen exchange rate report. Returned HTTP status: ' . $this->_curl->getStatus());
        }
        $response = $this->_curl->getBody();
        $raw_data = array_slice(explode("\n", $response), 5);
        return $raw_data;
    }


    protected function parseRateData($rates_raw_data, $date)
    {
        $parsed_rates = array();
        $rates = array_map('str_getcsv', $rates_raw_data);
        foreach ($rates as $rate) {
            if (count($rate) == 1) {
                continue;
            }
            if ($rate[0] != 'EUR') {
                continue;
            }
            $symbol = $rate[1];
            $conversion_rate = $rate[5];
            //$exponent = $rate[4]; unused
            if (isset($this->currencies[$symbol])) {
                $parsed_rates[$symbol] = array('day_date' => $date, 'currency' => $symbol, 'rate' => $this->adjustRate($conversion_rate, $symbol));
            }
        }
        return $parsed_rates;
    }


    protected function adjustRate($conversion_rate, $symbol)
    {
        $markup_factor = 1 - $this->currencies[$symbol];
        $rate = 1 / $conversion_rate;
        $rate = $rate * $markup_factor;
        return round(1 / $rate, 4);
    }

}
