<?php

namespace FiloBlu\ActiveCampaign\Cron;

use FiloBlu\ActiveCampaign\Api\Data\JobsInterface;
use FiloBlu\ActiveCampaign\Helper\Data;
use FiloBlu\ActiveCampaign\Model\Config\Source\DirectionType;
use FiloBlu\ActiveCampaign\Model\Config\Source\EntityType;
use FiloBlu\ActiveCampaign\Model\Config\Source\JobStatus;
use FiloBlu\ActiveCampaign\Model\Config\Source\Type;
use FiloBlu\ActiveCampaign\Model\JobManager as JobManagerModel;

/**
 * Class JobManager
 * @package FiloBlu\ActiveCampaign\Cron
 */
class JobManager
{
    /**
     * Constants
     */
    public const OPERATION = 'operation';
    public const OPERATION_PREPARE = 'prepare';
    public const OPERATION_PROCESS = 'process';
    public const ALLOWED_OPERATIONS = [self::OPERATION_PREPARE, self::OPERATION_PROCESS];

    public const EXPORT_TYPE = 'export_type';

    /**
     * @var JobManagerModel
     */
    protected $jobManager;
    /**
     * @var Data
     */
    protected $helperData;

    /**
     * JobManager constructor.
     * @param JobManagerModel $jobManager
     * @param Data $helperData
     */
    public function __construct(
        JobManagerModel $jobManager,
        Data $helperData
    ) {
        $this->jobManager = $jobManager;
        $this->helperData = $helperData;
    }

    /**
     * PREPARE SUBSCRIBER
     */
    public function prepareSubscriber()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PREPARE,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_EXPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_SUBSCRIBER,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }

    ///////////////////////////////////// PREPARE ///////////////////////////////////////////

    /**
     * @param $options
     * @return array
     */
    public function executeJob($options)
    {
        $status = '';
        $message = '';

        if (!$this->helperData->isEnabled()) {
            return ['info', 'All export are disabled by config'];
        }

        switch ($options[self::OPERATION]) {
            case self::OPERATION_PREPARE:
                list($status, $message) = $this->jobManager->setEntity($options[JobsInterface::ENTITY])->setExportType(
                        $options[self::EXPORT_TYPE]
                    )->setDirection($options[JobsInterface::DIRECTION])->setType(
                        $options[JobsInterface::TYPE]
                    )->setStatusToPrepare($options[JobsInterface::STATUS])->prepare();
                break;
            case self::OPERATION_PROCESS:
                list($status, $message) = $this->jobManager->setDirection(
                        $options[JobsInterface::DIRECTION]
                    )->setEntity($options[JobsInterface::ENTITY])->setStatusToProcess(
                        $options[JobsInterface::STATUS]
                    )->setType($options[JobsInterface::TYPE])->process();
                break;
        }

        return [$status, $message];
    }

    /**
     * PREPARE CUSTOMER
     */
    public function prepareCustomer()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PREPARE,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_EXPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_CUSTOMER,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }

    /**
     * PREPARE QUOTE
     */
    public function prepareQuote()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PREPARE,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_EXPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_QUOTE,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }

    /**
     * PREPARE ORDER
     */
    public function prepareOrder()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PREPARE,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_EXPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_ORDER,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }


    ///////////////////////////////////// PROCESS ///////////////////////////////////////////

    /**
     * PROCESS SUBSCRIBER
     */
    public function processSubscriber()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PROCESS,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_EXPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_SUBSCRIBER,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }

    /**
     * PROCESS CUSTOMER
     */
    public function processCustomer()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PROCESS,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_EXPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_CUSTOMER,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }

    /**
     * PROCESS QUOTE
     */
    public function processQuote()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PROCESS,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_EXPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_QUOTE,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }

    /**
     * PROCESS ORDER
     */
    public function processOrder()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PROCESS,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_EXPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_ORDER,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }

    /**
     * PROCESS SUBSCRIBER IMPORT
     */
    public function processSubscriberImport()
    {
        $options = [
            self::OPERATION          => self::OPERATION_PROCESS,
            JobsInterface::TYPE      => Type::TYPE_REST,
            JobsInterface::DIRECTION => DirectionType::DIRECTION_IMPORT,
            JobsInterface::ENTITY    => EntityType::ENTITY_SUBSCRIBER,
            self::EXPORT_TYPE        => JobsInterface::INCREMENTAL,
            JobsInterface::STATUS    => JobStatus::JOB_STATUS_WAITING
        ];
        $this->executeJob($options);
    }

}
